'use client';

import { useQuery } from 'react-query';

import {
	JobMinistryTag,
	Loader,
	TableReusableComponent,
} from '@/components/others/CommonComponents';

import { fetchEndpointData } from '@/components/utils/server-functions';
import { ApplicantsColumns } from '@/components/table/Columns';
import {
	Application,
	TApplicantStatus,
	TFilters,
	TMinistry,
} from '@/components/utils/types';
import { useQueryCustomWrapper } from '@/utils';
import { useState } from 'react';

export default function () {
	const [page, setCurrentPage] = useState(1);

	const [filters, setFilters] = useState<TFilters>({
		ministries: [],
		term: '',
	} as TFilters);

	const {
		isLoading: isMinistryLoading,
		isError: isMinistryError,
		data: ministries,
	} = useQuery({
		queryFn: useQueryCustomWrapper<TMinistry[]>,
		queryKey: [
			`ministry-data`,
			{
				url: `ministries`,
				qFunc: fetchEndpointData,
			},
		],
	});

	const { isLoading, isError, data } = useQuery({
		queryFn: useQueryCustomWrapper<Application[]>,
		queryKey: [
			`awarded_applicants-${filters.ministries.join('-')}`,
			{
				url: `applications`,
				qFunc: fetchEndpointData,
				options: {
					populate: {
						user: true,
						job: {
							populate: ['ministry', 'job_type'],
						},
					},

					filters: {
						status: TApplicantStatus.Awarded,
					},
					sort: 'createdAt:desc',
					pagination: {
						page: page,
					},
				},
			},
		],
	});

	return (
		<div className=' h-full flex flex-col pl-[15px] pr-[15px] pt-[16px] pb-[33px] gap-y-[32px] md:gap-y-0 md:gap-[20px] md:px-[16px]'>
			<section className='w-full flex flex-col gap-y-[24px] md:mb-2'>
				{/* table */}
				<section className='gap-y-[32px] md:gap-y-0 md:col-span-3 h-fit overflow-auto'>
					<div className='md:col-span-6'>
						{isMinistryLoading || isLoading ? (
							<div className='w-full h-[500px] bg-sidebar-bg-light dark:bg-small-card-bg'>
								<Loader />
							</div>
						) : isMinistryError || isError ? (
							<div className='w-full h-[500px] bg-sidebar-bg-light dark:bg-small-card-bg'>
								<Loader
									{...{
										title: 'Error loading data',
									}}
								/>
							</div>
						) : (
							<TableReusableComponent
								{...{
									title: (
										<div className='flex items-center gap-[8px]'>
											<p className='leading-[28px] text-table-title-color font-bold text-[20px] md:text-[24px] tracking-[-.5%]'>
												Awarded applicants
											</p>
											<JobMinistryTag
												{...{
													ministry_name: `${
														isLoading
															? 'Loading...'
															: data?.meta?.pagination.total
															? `${data?.meta?.pagination.total} Applicants`
															: ''
													} `,
													textClassName:
														'bg-applicant-colorbg text-applicant-colorText',
													className:
														'bg-applicant-colorbg rounded-[16px] px-[8px] py-[2px]',
													dotClass: 'hidden',
												}}
											/>
										</div>
									),
									columns: ApplicantsColumns,
									titleFilterInline: true,
									showPagination: true,
									pageCount: data?.meta?.pagination?.pageCount ?? 0,
									ministryFilterTitle: 'Filter applicants by ministry',
									pageSize: data?.meta.pagination.page || 0,
									onPageChange: (page) => {
										setCurrentPage(page);
									},
									filterData: {
										data: ministries?.data ?? [],
										setFilters,
										filters,
									},
									isSearchAtEnd: true,
									searchColumn: 'user',
									data: data?.data ?? [],
									filter: true,
								}}
							/>
						)}
					</div>
				</section>
			</section>
		</div>
	);
}
