'use client';

import { useQuery } from 'react-query';

import {
	Loader,
	TableReusableComponent,
} from '@/components/others/CommonComponents';

import { JobsColumns, actionsColumn } from '@/components/table/Columns';
import { useQueryCustomWrapper } from '@/utils';
import { fetchEndpointData } from '@/components/utils/server-functions';
import AlertDialogComponent from '@/components/reusables/AlertDialog';
import { useState } from 'react';
import { BASE_URL, JobOptionsEnum } from '@/components/utils/constants';
import { useRouter } from 'next/navigation';
import {
	ActionMapperType,
	JobStatusEnum,
	TDropdownHandlerprops,
	TFilters,
	TJob,
	TMinistry,
} from '@/components/utils/types';

export default function () {
	const [action, setAction] = useState<TDropdownHandlerprops>({
		id: undefined,
		label: '',
	});

	const [filters, setFilters] = useState<TFilters>({
		ministries: [],
		term: '',
	} as TFilters);

	const [page, setCurrentPage] = useState(1);

	const router = useRouter();

	const { isLoading, isError, data } = useQuery({
		queryFn: useQueryCustomWrapper<TJob[]>,
		queryKey: [
			`jobs-data`,
			{
				url: `jobs`,
				qFunc: fetchEndpointData,
				options: {
					sort: 'createdAt:desc',
					populate: '*',
					filters: {
						status: JobStatusEnum.Closed,
						...(filters.ministries.length > 0
							? { ministry: { name: { $in: filters.ministries } } }
							: {}),
						...(filters.term ? { title: { $containsi: filters.term } } : {}),
					},
					pagination: {
						page: page,
					},
				},
			},
		],
		enabled: !!page,
	});

	const {
		isLoading: isMinistryLoading,
		isError: isMinistryError,
		data: ministries,
	} = useQuery({
		queryFn: useQueryCustomWrapper<TMinistry[]>,
		queryKey: [
			`ministry-data`,
			{
				url: `ministries`,
				qFunc: fetchEndpointData,
			},
		],
	});

	const ActionsHandlerMapping: ActionMapperType = {
		onChangeHandler: ({ id, label }: TDropdownHandlerprops) =>
			setAction({ label, id }),
		data: [
			{
				label: JobOptionsEnum['Edit Job'],
			},

			{
				label: JobOptionsEnum['Close Job'],
			},
		],
	};

	return (
		<div className=' h-full flex flex-col pl-[15px] pr-[15px] pt-[16px] pb-[33px] gap-y-[32px] md:gap-y-0 md:gap-[20px] md:px-[16px]'>
			<section className='w-full flex flex-col gap-y-[24px] md:mb-2'>
				{/* table */}
				<section className='gap-y-[32px] md:gap-y-0 md:col-span-3 h-fit overflow-auto'>
					<div className='md:col-span-6'>
						{isMinistryLoading ? (
							<div className='md:col-span-6 rounded-[12px] bg-white shadow-sidebarShadow min-h-[500px] w-full flex items-center justify-center'>
								<Loader />
							</div>
						) : isMinistryError ? (
							<div className='md:col-span-6 rounded-[12px] bg-white shadow-sidebarShadow min-h-[500px] w-full flex items-center justify-center'>
								<Loader
									{...{
										title: 'Error loading data',
									}}
								/>
							</div>
						) : (
							<TableReusableComponent
								{...{
									data: data?.data ?? [],
									columns: [
										...JobsColumns,
										actionsColumn({ ActionsHandlerMapping }),
									],
									pageCount: data?.meta?.pagination?.pageCount ?? 0,
									onPageChange: (page) => setCurrentPage(page),
									pageSize: data?.meta.pagination.page || 0,
									ministryFilterTitle: 'Filter jobs by ministry',
									titleFilterInline: true,
									showPagination: true,
									title: 'Closed Jobs',
									isLoading,
									isError,
									filterData: {
										data: ministries?.data ?? [],
										setFilters,
										filters,
									},
									headerBtn: {
										action: () => router.push(`${BASE_URL}/jobs/create`),
										text: 'Add Job Posting',
										present: true,
									},
									filter: true,
								}}
							/>
						)}

						{action.label === JobOptionsEnum['Close Job'] && (
							<AlertDialogComponent
								{...{
									onClicked: () =>
										// send request to backend to proceed the action
										setAction({ label: '', id: undefined }),
									description:
										'This action cannot be undone. This will permanently delete this record from the server.',
									title: 'Are you sure?',
								}}
							/>
						)}

						{action.label === JobOptionsEnum['Edit Job'] &&
							(() => {
								router.push(`${BASE_URL}/jobs/edit/${action.id}`);
								return null;
							})()}
					</div>
				</section>
			</section>
		</div>
	);
}
