'use client';
import { useQuery } from 'react-query';
import { useState } from 'react';

import {
	LineGraphReusableComponent,
	TableReusableComponent,
	JobMinistryTag,
	Loader,
} from '@/components/others/CommonComponents';
import {
	TDashboardData,
	Application,
	TFilters,
	TMinistry,
	TApplicantStatus,
} from '@/components/utils/types';
import { ApplicantsIcon, TickIcon, WorkIcon } from '@/components/icons';
import { fetchEndpointData } from '@/components/utils/server-functions';
import { ApplicantsColumns } from '@/components/table/Columns';
import Upcoming from '@/components/reusables/Upcoming';
import { Card } from '@/components/reusables/Cards';
import { useQueryCustomWrapper } from '@/utils';

export default function dashboard() {
	const [filters, setFilters] = useState<TFilters>({
		ministries: [],
		term: '',
	} as TFilters);

	const [page, setCurrentPage] = useState(1);

	const { data, isLoading, isError } = useQuery({
		queryFn: useQueryCustomWrapper<TDashboardData>,
		queryKey: [
			`overview-applicant-summaries`,
			{
				url: `dashboard`,
				qFunc: fetchEndpointData,
				options: {
					// sort: 'name:asc',
					// populate: '*',
				},
			},
		],
	});

	const {
		isLoading: isMinistryLoading,
		isError: isMinistryError,
		data: ministries,
	} = useQuery({
		queryFn: useQueryCustomWrapper<TMinistry[]>,
		queryKey: [
			`ministry-data`,
			{
				url: `ministries`,
				qFunc: fetchEndpointData,
			},
		],
	});

	const {
		isLoading: isJobsLoading,
		isError: isJobsError,
		data: jobs,
	} = useQuery({
		queryFn: useQueryCustomWrapper<Application[]>,
		queryKey: [
			`recent-applications-${filters.ministries.join('-')}`,
			{
				url: 'applications',
				qFunc: fetchEndpointData,
				options: {
					populate: {
						user: true,
						job: {
							populate: { ministry: true, job_type: true },
						},
					},
					filters: {
						status: TApplicantStatus.New,
						job: {
							...(filters.ministries.length > 0
								? { ministry: { name: { $in: filters.ministries } } }
								: {}),
							...(filters.term ? { title: { $containsi: filters.term } } : {}),
						},
					},
					sort: 'createdAt:desc',
					pagination: {
						page: page,
					},
				},
			},
		],
		enabled: !!page,
	});

	return (
		<div className='mb-[10px] p-[10px] h-full'>
			<div className='pb-[40px]'>
				{/* first part  */}
				<div className=' h-auto flex gap-[12px] w-full'>
					<div className='flex-[5] bg-sidebar-bg-light dark:bg-small-card-bg rounded-[8px] h-[480px] p-[16px] shadow-cardBoxShadow dark:shadow-cardBoxShadowDark'>
						<LineGraphReusableComponent
							{...{
								graphData: data?.data.applications ?? [],
								yAxisLabel: 'No of Applicants',
								title: 'Job Applicantions',
								xAxisLabel: 'Month',
								filterBy: false,
								isLoading,
								isError,
							}}
						/>
					</div>

					{/* upcoming events and calendar section  */}
					<div className='flex-[2] flex flex-col gap-[30px]'>
						<Upcoming
							{...{
								applications: jobs?.data ?? [],
								isLoading: isJobsLoading,
								isError: isJobsError,
							}}
						/>
					</div>
					{/* end of last section  */}
				</div>
				{/* end of first part  */}
			</div>

			{/* cards section  */}
			{/* text-body dark:text-sidebar-bg-light */}
			<div className='flex flex-col gap-[12px] justify-center md:flex-row md:justify-between my-[20px] md:flex-wrap'>
				<div className='w-full grow basis-[300px] max-w-[450px] h-[156px] dark:text-sidebar-bg-light text-body bg-sidebar-bg-light dark:bg-small-card-bg p-[6px] rounded-[6px] flex justify-between relative shadow-cardBoxShadow dark:shadow-cardBoxShadowDark'>
					{isLoading ? (
						<div className='w-full h-full'>
							<Loader />
						</div>
					) : isError ? (
						<div className='w-full h-full'>
							<Loader {...{ title: 'Error loading data' }} />
						</div>
					) : (
						<Card
							{...{
								lastMonth: data?.data?.applicationsLastMonth! ?? 0,
								thisMonth: data?.data?.applicationsThisMonth! ?? 0,
								trend: data?.data.applicationTrend,
								title: 'Application received',
								Icon: ApplicantsIcon,
							}}
						/>
					)}
				</div>

				<div className='w-full grow basis-[300px] max-w-[450px] h-[156px] dark:text-sidebar-bg-light text-body bg-sidebar-bg-light dark:bg-small-card-bg p-[6px] rounded-[6px] flex justify-between relative shadow-cardBoxShadow dark:shadow-cardBoxShadowDark'>
					{isLoading ? (
						<div className='w-full h-full'>
							<Loader />
						</div>
					) : isError ? (
						<div className='w-full h-full'>
							<Loader {...{ title: 'Error loading data' }} />
						</div>
					) : (
						<Card
							{...{
								lastMonth: data?.data?.jobsLastMonth! ?? 0,
								thisMonth: data?.data?.jobsThisMonth! ?? 0,
								trend: data?.data.jobTrend,
								title: 'Jobs posted',
								Icon: WorkIcon,
							}}
						/>
					)}
				</div>

				<div className='w-full grow basis-[300px] max-w-[450px] h-[156px] dark:text-sidebar-bg-light text-body bg-sidebar-bg-light dark:bg-small-card-bg p-[6px] rounded-[6px] flex justify-between relative shadow-cardBoxShadow dark:shadow-cardBoxShadowDark'>
					{isLoading ? (
						<div className='w-full h-full'>
							<Loader />
						</div>
					) : isError ? (
						<div className='w-full h-full'>
							<Loader {...{ title: 'Error loading data' }} />
						</div>
					) : (
						<Card
							{...{
								lastMonth: data?.data?.applicationsLastMonth! ?? 0,
								thisMonth: data?.data?.applicationsThisMonth! ?? 0,
								trend: data?.data?.applicationTrend,
								title: 'Hired applicants',
								Icon: TickIcon,
							}}
						/>
					)}
				</div>
			</div>

			<div className=' h-full flex flex-col pl-[15px] pr-[15px] pt-[16px] pb-[33px] gap-y-[32px] md:gap-y-0 md:gap-[20px] md:px-[16px]'>
				<section className='w-full flex flex-col gap-y-[24px] md:mb-2'>
					{/* table */}
					<section className='gap-y-[32px] md:gap-y-0 md:col-span-3 h-fit overflow-auto'>
						<div className='md:col-span-6'>
							{isMinistryLoading ? (
								<div className='w-full h-[500px] bg-sidebar-bg-light dark:bg-small-card-bg'>
									<Loader />
								</div>
							) : isMinistryError ? (
								<div className='w-full h-[500px] bg-sidebar-bg-light dark:bg-small-card-bg'>
									<Loader
										{...{
											title: 'Loading data...',
										}}
									/>
								</div>
							) : (
								<TableReusableComponent
									{...{
										title: (
											<div className='flex items-center gap-[8px]'>
												<p className='leading-[28px] text-table-title-color font-bold text-[20px] md:text-[24px] tracking-[-.5%]'>
													Recent applications
												</p>
												<JobMinistryTag
													{...{
														textClassName:
															'bg-applicant-colorbg text-applicant-colorText',
														ministry_name: `${
															isJobsLoading
																? 'Loading...'
																: jobs?.meta?.pagination.total
																? `${jobs?.meta?.pagination.total} Applicants`
																: ''
														} `,
														className:
															'bg-applicant-colorbg rounded-[16px] px-[8px] py-[2px]',
														dotClass: 'hidden',
													}}
												/>
											</div>
										),
										pageCount: jobs?.meta?.pagination?.pageCount ?? 0,
										pageSize: jobs?.meta.pagination.page || 0,
										filterData: {
											data: ministries?.data ?? [],
											setFilters,
											filters,
										},
										onPageChange: (page) => setCurrentPage(page),
										ministryFilterTitle: 'Filter applicants by ministry',
										columns: ApplicantsColumns,
										titleFilterInline: true,
										showPagination: true,
										isLoading: isJobsLoading,
										isSearchAtEnd: true,
										isError: isJobsError,
										data: jobs?.data!,
										pagination: true,
										filter: true,
									}}
								/>
							)}
						</div>
					</section>
				</section>
			</div>
		</div>
	);
}
