'use client';

import { useQuery } from 'react-query';
import { useState } from 'react';

import { StaffColumns, userActionsColumn } from '@/components/table/Columns';
import {
	fetchEndpointData,
	updateResourceEndpointData,
} from '@/components/utils/server-functions';
import AlertDialogComponent from '@/components/reusables/AlertDialog';
import AddJobsBtn from '@/components/reusables/Download';
import {
	BASE_URL,
	URL_SEARCH_PARAMS,
	UserOptionsEnum,
} from '@/components/utils/constants';
import { useQueryCustomWrapper } from '@/utils';
import {
	TableReusableComponent,
	JobMinistryTag,
	Loader,
} from '@/components/others/CommonComponents';
import { AddUserIcon } from '@/components/icons';
import {
	ActionMapperType,
	TDropdownHandlerprops,
	TFilters,
	TMinistry,
	TUSER,
} from '@/components/utils/types';
import { usePathname, useRouter } from 'next/navigation';
import { ToastContainer, toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

export default function StaffPage() {
	const [action, setAction] = useState<TDropdownHandlerprops>({
		id: undefined,
		label: '',
	});

	const [filters, setFilters] = useState<TFilters>({
		ministries: [],
		term: '',
	} as TFilters);

	const {
		isLoading: isMinistryLoading,
		isError: isMinistryError,
		data: ministries,
	} = useQuery({
		queryFn: useQueryCustomWrapper<TMinistry[]>,
		queryKey: [
			`ministry-data`,
			{
				url: `ministries`,
				qFunc: fetchEndpointData,
			},
		],
	});

	const [page, setCurrentPage] = useState(1);
	const pathname = usePathname();
	const router = useRouter();

	const { isLoading, isError, data, refetch } = useQuery({
		queryFn: useQueryCustomWrapper<TUSER[]>,
		queryKey: [
			`staff-data`,
			{
				url: `auth/users`,
				qFunc: fetchEndpointData,
				options: {
					sort: 'createdAt:desc',
					populate: ['ministry'],
					filters: {
						...(filters.ministries.length > 0
							? { ministry: { name: { $in: filters.ministries } } }
							: {}),

						...(filters.term
							? {
									$or: [
										{
											username: { $containsi: filters.term },
										},
										{
											firstname: { $containsi: filters.term },
										},
										{
											lastname: { $containsi: filters.term },
										},
									],
							  }
							: {}),
					},
					pagination: {
						page: page,
					},
				},
			},
		],
		enabled: !!page,
	});

	const ActionsHandlerMapping: ActionMapperType = {
		onChangeHandler: ({ label, id }: TDropdownHandlerprops) =>
			setAction({ label, id }),
		data: [
			// {
			// 	label: UserOptionsEnum.Edit,
			// },

			{
				label: UserOptionsEnum.Suspend,
			},

			{
				label: UserOptionsEnum.Activate,
			},
		],
	};

	const handleDataUpdate = ({
		data,
		id,
	}: {
		data: { [key: string]: string | number | boolean };
		id: string | number;
	}) => {
		console.log('Called', data);

		updateResourceEndpointData({
			url: `users/${id}`,
			method: 'PUT',
			data,
		}).then(({ data: responseData, err }) => {
			if (err) {
				if (err.status === 400)
					toast.error('Bad request', { position: 'top-right' });
				else if (err.status === 401)
					router.push(
						`/?${URL_SEARCH_PARAMS.redirect}=${encodeURIComponent(pathname)}`
					);
				else if (err.status === 403)
					toast.error('Permission denied', {
						position: 'top-right',
					});
				else
					toast.error('Something went wrong', {
						position: 'top-right',
					});
			} else if (responseData) {
				refetch();
				toast.success('Information saved successfully.', {
					position: 'top-right',
				});
			} else
				toast.error('Something went wrong.', {
					position: 'top-right',
				});
		});
	};

	return (
		<div className='px-[16px]'>
			<div className='flex justify-between mx-[16px] mt-[20px]'>
				<p className='tracking-[-.5%] leading-[28px] text-[24px] font-bold text-table-title-color'>
					Kitui County
				</p>

				<div className='w-fit'>
					<AddJobsBtn
						{...{
							action: () => router.push(`${BASE_URL}/staff/create`),
							Icon: AddUserIcon,
							text: 'Add User',
							present: true,
						}}
					/>
				</div>
			</div>

			<div className=' h-full flex flex-col pl-[15px] pr-[15px] pt-[16px] pb-[33px] gap-y-[32px] md:gap-y-0 md:gap-[20px] md:px-[16px]'>
				<section className='w-full flex flex-col gap-y-[24px] md:mb-2'>
					{/* table */}
					<section className='gap-y-[32px] md:gap-y-0 md:col-span-3 h-fit overflow-auto'>
						<div className='md:col-span-6'>
							{isMinistryLoading ? (
								<div className='w-full h-[500px] bg-sidebar-bg-light dark:bg-small-card-bg flex items-center justify-center'>
									<Loader />
								</div>
							) : isMinistryError ? (
								<div className='w-full h-[500px] bg-sidebar-bg-light dark:bg-small-card-bg flex items-center justify-center'>
									<Loader
										{...{
											title: 'Error fetching data',
										}}
									/>
								</div>
							) : (
								<TableReusableComponent
									{...{
										title: (
											<div className='flex items-center gap-[8px]'>
												<p className='leading-[28px] text-table-title-color font-bold text-[20px] md:text-[24px] tracking-[-.5%]'>
													System administrators
												</p>
												<JobMinistryTag
													{...{
														textClassName:
															'bg-applicant-colorbg text-applicant-colorText',
														ministry_name: `${
															isLoading
																? 'Loading...'
																: data?.meta?.pagination.total
																? `${data?.meta?.pagination.total} Admins`
																: ''
														} `,
														className:
															'bg-applicant-colorbg rounded-[16px] px-[8px] py-[2px]',
														dotClass: 'hidden',
													}}
												/>
											</div>
										),
										data: data?.data ?? [],
										columns: [
											...StaffColumns,
											userActionsColumn({ ActionsHandlerMapping }),
										],
										onPageChange: (page) => setCurrentPage(page),
										pageCount: data?.meta?.pagination?.pageCount ?? 0,
										ministryFilterTitle: 'Filter staff by ministry',
										pageSize: data?.meta.pagination.page || 0,
										titleFilterInline: true,
										showPagination: true,
										filterData: {
											data: ministries?.data ?? [],
											setFilters,
											filters,
										},
										isSearchAtEnd: true,
										filter: true,
										isLoading,
										isError,
									}}
								/>
							)}
						</div>
					</section>
				</section>

				{action.label === UserOptionsEnum.Suspend && (
					<AlertDialogComponent
						{...{
							onClicked: () => {
								// send request to backend to proceed the action
								if (action.id)
									handleDataUpdate({ data: { blocked: true }, id: action.id });

								setAction({ id: undefined, label: '' });
							},
							description: 'This action will deny access to this user',
							title: 'Are you sure?',
						}}
					/>
				)}

				{action.label === UserOptionsEnum.Edit &&
					(() => {
						router.push(`${BASE_URL}/staff/edit/${action.id}`);

						return true;
					})()}

				{action.label === UserOptionsEnum.Activate && (
					<AlertDialogComponent
						{...{
							onClicked: () => {
								// send request to backend to proceed the action
								if (action.id)
									handleDataUpdate({ data: { blocked: false }, id: action.id });

								setAction({ id: undefined, label: '' });
							},
							description: 'This action will unblock this user',
							title: 'Are you sure?',
						}}
					/>
				)}

				{action.label === UserOptionsEnum.Edit &&
					(() => {
						router.push(`${BASE_URL}/staff/edit/${action.id}`);

						return true;
					})()}
			</div>

			<ToastContainer />
		</div>
	);
}
