'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { useState } from 'react';
import { z } from 'zod';

import {
	Form,
	FormControl,
	FormField,
	FormItem,
	FormLabel,
	FormMessage,
} from '../ui/form';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import Link from 'next/link';
import { Checkbox } from '../ui/checkbox';
import {
	AuthenticateUser,
	ResetPasswordHandler,
} from '../utils/server-functions';
import { useRouter } from 'next/navigation';

export const AuthScreen = () => {
	const [authLoading, setAuthLoading] = useState<boolean>(false);
	const [authError, setAuthError] = useState<string | null>(null);
	const router = useRouter();

	const FormSchema = z.object({
		password: z.string().min(2, {
			message: 'Name must be at least 2 characters.',
		}),

		identifier: z
			.string()
			.email({ message: 'Email field must contain a valid email' }),
	});

	const form = useForm<z.infer<typeof FormSchema>>({
		resolver: zodResolver(FormSchema),
		defaultValues: {
			password: '',
			identifier: '',
		},
	});

	function onSubmit(data: z.infer<typeof FormSchema>) {
		setAuthLoading(true);
		setAuthError('');
		const res = AuthenticateUser({ ...data });

		res
			.then(({ err, data }) => {
				if (err) setAuthError(err);
				else router.push('/dashboard');
			})
			.catch(() =>
				setAuthError("Username and password combination didn't match")
			)
			.finally(() => setAuthLoading(false));
	}

	return (
		<div className='py-[20px] mx-auto flex flex-col gap-[24px] justify-center items-center w-fit h-full'>
			<div className='w-fit flex items-center justify-center'>
				<img
					src={'/images/logo/logo.png'}
					className='w-[100px] h-[100px]'
					alt='Logo'
				/>
			</div>

			<div className='w-full flex gap-[40px] p-[32px] flex-wrap justify-center items-center flex-col rounded-[16px] bg-white shadow-page404Shadow'>
				{authError ? (
					<div className='text-red-700 text-center w-fit mx-auto'>
						<p>{authError}</p>
					</div>
				) : null}

				<Form {...form}>
					<form
						onSubmit={form.handleSubmit(onSubmit)}
						className='w-full space-y-6 flex items-center justify-center'>
						<div className='flex flex-col gap-[24px] px-[32px] w-fit'>
							{/* form header  */}
							<div className='flex gap-[12px] flex-col'>
								<p className='font-bold text-[30px] leading-[36px] tracking-[.75%] text-title-text-color'>
									Log In
								</p>
								<p className='text-gray-body-text leading-[24px] text-[16px] font-normal'>
									Welcome back! Please enter your details.
								</p>
							</div>

							<div className='flex-[1]'>
								<FormField
									control={form.control}
									name='identifier'
									render={({ field }) => (
										<FormItem>
											<FormLabel className='text-title-text-color'>
												Email
											</FormLabel>
											<FormControl>
												<Input
													placeholder={'Email address'}
													{...field}
													className='rounded-[6px] px-[12px] py-[14px] border border-gray-300 bg-login-screen-text-color text-black leading-[24px] text-[14px] font-normal flex items-center'
												/>
											</FormControl>
											<FormMessage />
										</FormItem>
									)}
								/>
							</div>

							<div className='flex-[1]'>
								<FormField
									control={form.control}
									name={'password'}
									render={({ field }) => (
										<FormItem>
											<FormLabel className='text-title-text-color'>
												Password
											</FormLabel>
											<FormControl>
												<Input
													placeholder={'Password'}
													{...field}
													className='rounded-[6px] px-[12px] py-[14px] border border-gray-300 bg-login-screen-text-color text-black leading-[24px] text-[14px] font-normal flex items-center'
													type={'password'}
												/>
											</FormControl>
											<FormMessage />
										</FormItem>
									)}
								/>
							</div>

							{/* forgot pass section  */}
							<div className='flex flex-wrap gap-[24px] py-[0px] text-filter-stroke-color justify-between'>
								<div className='flex gap-[8px]'>
									<Checkbox id='remember' className='' />
									<label
										htmlFor='remember'
										className='font-medium text-[14px] leading-[20px] text-filter-stroke-color selection:bg-inherit'>
										Remember me
									</label>
								</div>
								<div className='w-fit'>
									<Link
										href={'/reset-password'}
										className='text-[14px] font-bold leading-[20px] text-purple-800'>
										Forgot password
									</Link>
								</div>
							</div>

							<div className='w-full justify-center flex-col flex my-[8px] gap-[32px]'>
								<Button
									className='rounded-[8px] bg-dev-accent hover:bg-dev-accent text-white border-dev-accent border px-[10px] py-[16px] gap-[8px] shadow-btnBoxShadow items-center justify-center w-full leading-[24px] text-[16px] font-bold'
									type='submit'
									{...(authLoading ? { disabled: true } : {})}>
									{authLoading ? 'Submitting...' : 'Sign in'}
								</Button>
							</div>
						</div>
					</form>
				</Form>
			</div>
		</div>
	);
};

export const ResetPassword = () => {
	const [authLoading, setAuthLoading] = useState<boolean>(false);
	const [authError, setAuthError] = useState<string | null>(null);
	const [success, setSuccess] = useState('');

	const FormSchema = z.object({
		email: z
			.string()
			.email({ message: 'Email field must contain a valid email' }),
	});

	const form = useForm<z.infer<typeof FormSchema>>({
		resolver: zodResolver(FormSchema),
		defaultValues: {
			email: '',
		},
	});

	function onSubmit(data: z.infer<typeof FormSchema>) {
		setAuthLoading(true);
		setAuthError('');

		ResetPasswordHandler({
			url: '/auth/forgot-password',
			data,
		})
			.then(({ data: res, err }) => {
				if (err)
					if (err.status === 400)
						err.details.errors.map(({ message }) =>
							form.setError('email', {
								message,
							})
						);
					else setAuthError('Something went wrong');
				if (data) setSuccess('Check your email for the reset password link');
			})
			.finally(() => setAuthLoading(false));
	}

	return (
		<div className='py-[20px] mx-auto flex flex-col gap-[24px] justify-center items-center w-fit h-full'>
			<div className='w-fit flex items-center justify-center'>
				<img
					src={'/images/logo/logo.png'}
					className='w-[100px] h-[100px]'
					alt='Logo'
				/>
			</div>

			<div className='w-full flex gap-[40px] p-[32px] flex-wrap justify-center items-center flex-col rounded-[16px] bg-white shadow-page404Shadow max-w-[514px]'>
				<Form {...form}>
					<form
						onSubmit={form.handleSubmit(onSubmit)}
						className='w-full space-y-6 flex items-center justify-center'>
						<div className='flex flex-col gap-[24px] px-[32px] w-fit'>
							{/* form header  */}
							<div className='flex gap-[12px] flex-col'>
								<p className='font-bold text-[30px] leading-[36px] tracking-[.75%] text-title-text-color'>
									Reset your password
								</p>
								<p className='text-gray-body-text leading-[24px] text-[16px] font-normal text-wrap'>
									Enter your email address and we will send you a link to reset
									your password.
								</p>
							</div>

							<div className='flex flex-col gap-[24px] px-[16px] w-fit'>
								{authError ? (
									<div className='text-red-700 text-center w-fit mx-auto'>
										<p>{authError}</p>
									</div>
								) : null}
								{success ? (
									<div className='text-green-700 text-center w-fit mx-auto'>
										<p>{success}</p>
									</div>
								) : null}
							</div>

							<div className='flex-[1]'>
								<FormField
									control={form.control}
									name='email'
									render={({ field }) => (
										<FormItem>
											<FormLabel className='text-title-text-color'>
												Email
											</FormLabel>
											<FormControl>
												<Input
													placeholder={'Email address'}
													{...field}
													{...(success ? { disabled: true } : {})}
													className='rounded-[6px] px-[12px] py-[14px] border border-gray-300 bg-login-screen-text-color text-black leading-[24px] text-[14px] font-normal flex items-center'
												/>
											</FormControl>
											<FormMessage />
										</FormItem>
									)}
								/>
							</div>

							{/* forgot pass section  */}
							<div className='flex flex-wrap gap-[24px] py-[0px] text-filter-stroke-color justify-between'>
								<div className='w-fit flex gap-[2px] items-center'>
									<p className=''>Back to login</p>
									<Link
										href={'/'}
										className='text-[14px] font-bold leading-[20px] text-purple-800'>
										Signin
									</Link>
								</div>
							</div>

							<div className='w-full justify-center flex-col flex my-[8px] gap-[32px]'>
								<Button
									className='rounded-[8px] bg-dev-accent hover:bg-dev-accent text-white border-dev-accent border px-[10px] py-[16px] gap-[8px] shadow-btnBoxShadow items-center justify-center w-full leading-[24px] text-[16px] font-bold'
									type='submit'
									{...(authLoading || success ? { disabled: true } : {})}>
									{authLoading ? 'Submitting...' : 'Sign in'}
								</Button>
							</div>
						</div>
					</form>
				</Form>
			</div>
		</div>
	);
};

export default AuthScreen;
