'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { usePathname, useRouter } from 'next/navigation';
import { useForm } from 'react-hook-form';
import dynamic from 'next/dynamic';
import { format } from 'date-fns';
import {
	ArrowLeftIcon,
	ArrowRightIcon,
	CalendarIcon,
} from '@radix-ui/react-icons';
import { z } from 'zod';

import { Loader, UploadFileCard } from '@/components/others/CommonComponents';
import { Calendar } from '@/components/ui/calendar';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
	FormControl,
	FormMessage,
	FormItem,
	FormLabel,
	FormField,
	Form,
} from '@/components/ui/form';
import {
	Popover,
	PopoverContent,
	PopoverTrigger,
} from '@/components/ui/popover';
import {
	Select,
	SelectContent,
	SelectItem,
	SelectTrigger,
	SelectValue,
} from '@/components/ui/select';
import { BASE_URL, URL_SEARCH_PARAMS } from '@/components/utils/constants';
import { cn } from '@/lib/utils';
import {
	JobStatusEnum,
	Role,
	TCreateEditJobProps,
} from '@/components/utils/types';
import {
	createResourceEndpointData,
	deleteResourceEndpointData,
	uploadResourceEndpointData,
} from '../utils/server-functions';
import { RadioGroup, RadioGroupItem } from '../ui/radio-group';
import { useState } from 'react';
import { Textarea } from '../ui/textarea';
import { ToastContainer, toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

const RichTexEditor = dynamic(() => import('@/components/editor/RichText'), {
	ssr: false,
	loading: () => (
		<div className='dark:text-white w-[98%] h-[70px] selection:bg-inherit rounded-[12px] bg-sidebar-bg-light dark:bg-sidebar-bg text-body-bg dark:text-foundation flex items-center justify-self-center py-[10px] ml-4 mr-2'>
			<Loader
				{...{
					title: 'Loading...',
				}}
			/>
		</div>
	),
});

export const CreateUpdateJob = ({
	closed = false,
	successBtn,
	jobTypes,
	ministries,
	titleEnd,
	title,
	job,
}: TCreateEditJobProps) => {
	const [errMessage, setErrMsg] = useState<string | null>(null);
	const [selectedFiles, setSelectedFile] = useState<File[]>([]);
	const [loading, setLoading] = useState(false);
	const pathname = usePathname();
	const router = useRouter();

	const FormSchema = z.object({
		title: z.string().min(2, {
			message: 'Job title must be at least 2 characters.',
		}),

		application_start: z.date(),

		application_end: z.date(),

		description: z.string().min(6, {
			message: 'Job description must be at least 6 characters.',
		}),

		job_type: z.string().min(0, { message: 'Job type must be filled' }),

		ministry: z.string().min(0, {
			message: 'Ministry must be a valid field',
		}),

		location: z.string().min(2, {
			message: 'location must be field',
		}),

		about_job: z.string().min(2, {
			message: 'location must be field',
		}),
	});

	const form = useForm<z.infer<typeof FormSchema>>({
		resolver: zodResolver(FormSchema),
		defaultValues: {
			application_start: job?.application_start
				? new Date(job?.application_start)
				: new Date(),
			application_end: job?.application_end
				? new Date(job?.application_end)
				: new Date(),
			job_type: job?.job_type?.id ? `${job?.job_type?.id}` : '',
			ministry: job?.ministry?.id ? `${job?.ministry?.id}` : '',
			description: job?.description ?? '',
			about_job: job?.about_job ?? '',
			location: job?.location ?? '',
			title: job?.title ?? '',
		},
	});

	function onSubmit(data: z.infer<typeof FormSchema>) {
		const frmData = new FormData();

		selectedFiles.map((file) => frmData.append('files.files', file));

		frmData.append('data', `${JSON.stringify(data)}`);
		setLoading(true);

		uploadResourceEndpointData({
			method: successBtn.subDetails.method,
			url: successBtn.subDetails.url,
			data: frmData,
			...(job ? { id: job.id } : {}),
		})
			.then(({ data, err }) => {
				if (err) {
					if (err.status === 400)
						err.details.errors.map(({ path: [field_name], message, name }) =>
							//@ts-ignore
							form.setError(field_name, {
								message: message.replace(field_name, 'This field '),
							})
						);
					else if (err.status === 401)
						router.push(
							`/?${URL_SEARCH_PARAMS.redirect}=${encodeURIComponent(pathname)}`
						);
					else if (err.status === 403) {
						setErrMsg('Permission denied');
						toast.error('Permission denied', {
							position: 'top-right',
						});
					} else {
						setErrMsg('Something went wrong');
						toast.error('Something went wrong', {
							position: 'top-right',
						});
					}
				} else {
					form.reset();
					toast.success(
						`${
							successBtn.subDetails.method === 'POST'
								? 'Job Created'
								: 'Job Updated'
						}`,
						{
							position: 'top-right',
						}
					);
				}
			})
			.finally(() => setLoading(false));
	}

	const handleDeleteItem = (file: File) =>
		setSelectedFile(
			selectedFiles.filter((currItem) => {
				return currItem !== file;
			})
		);

	const handleDeleteRemoteItem = ({ id }: { id: string | number }) => {
		deleteResourceEndpointData({
			url: `upload/files/${id}`,
			data: {},
		})
			.then(({ err }) => {
				if (err) {
					if (err.status === 400)
						err.details.errors.map(({ path, message, name }) => {
							const [field_name, ...rest] = path;
							//@ts-ignore
							form.setError(field_name, {
								message: message.replace(field_name, 'This field '),
							});
						});
					else if (err.status === 401)
						router.push(
							`/?${URL_SEARCH_PARAMS.redirect}=${encodeURIComponent(pathname)}`
						);
					else if (err.status === 403) {
						setErrMsg('Permission denied');
						toast.error('Permission denied', {
							position: 'top-right',
						});
					} else {
						setErrMsg('Something went wrong');
						toast.error('Something went wrong', {
							position: 'top-right',
						});
					}
				} else
					toast.success('Item Deleted', {
						position: 'top-right',
					});
			})
			.finally(() => setLoading(false));
	};

	return (
		<div className='w-full py-[20px]'>
			<div className='flex justify-between items-center'>
				<p className='font-bold leading-[28px] text-[24px] text-title-text-color'>
					{title}
				</p>
				{job?.status === JobStatusEnum.Closed ? (
					<div className='text-[16px job-closed-text-color w-fit p-[10px] rounded-[10px] bg-job-closed-bg-color'>
						This job has been closed
					</div>
				) : (
					<div className='flex items-center gap-[6px]'>
						{titleEnd ? titleEnd : null}
					</div>
				)}
			</div>

			<div className='w-fit mx-auto'>
				<p className='text-red-500'>{errMessage}</p>
			</div>

			<div className='w-full h-fit flex gap-[20px] flex-wrap'>
				<Form {...form}>
					<form
						onSubmit={form.handleSubmit(onSubmit)}
						className='w-full space-y-6'>
						<div className='flex gap-[24px]'>
							<div className='flex-[1] flex flex-col gap-[24px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'title'}
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
													Job title
												</FormLabel>
												<FormControl>
													<Input
														placeholder='Health Office'
														{...field}
														// {...(job?.status === JobStatusEnum.Closed
														// 	? { disabled: true }
														// 	: {})}
														className='w-full'
													/>
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className='flex  gap-[24px]'>
									<div className='flex-[1]'>
										<FormField
											control={form.control}
											name={'application_start'}
											render={({ field }) => (
												<FormItem>
													<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
														Recruitment start date
													</FormLabel>
													<Popover>
														<PopoverTrigger asChild>
															<FormControl>
																<Button
																	variant={'outline'}
																	className={cn(
																		'w-[240px] pl-3 text-left font-normal',
																		!field.value && 'text-muted-foreground'
																	)}>
																	{field.value ? (
																		format(field.value, 'PPP')
																	) : (
																		<span>Pick a date</span>
																	)}
																	<CalendarIcon className='ml-auto h-4 w-4 opacity-50' />
																</Button>
															</FormControl>
														</PopoverTrigger>
														<PopoverContent
															className='w-auto p-0'
															align='start'>
															<Calendar
																mode='single'
																selected={field.value}
																onSelect={field.onChange}
																disabled={(date) =>
																	date < new Date('1900-01-01')
																}
																initialFocus
															/>
														</PopoverContent>
													</Popover>
													<FormMessage />
												</FormItem>
											)}
										/>
									</div>

									<div className='flex-[1]'>
										<FormField
											control={form.control}
											name={'application_end'}
											render={({ field }) => (
												<FormItem>
													<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
														Recruitment end date
													</FormLabel>
													<Popover>
														<PopoverTrigger asChild>
															<FormControl>
																<Button
																	variant={'outline'}
																	className={cn(
																		'w-[240px] pl-3 text-left font-normal',
																		!field.value && 'text-muted-foreground'
																	)}>
																	{field.value ? (
																		format(field.value, 'PPP')
																	) : (
																		<span>Pick a date</span>
																	)}
																	<CalendarIcon className='ml-auto h-4 w-4 opacity-50' />
																</Button>
															</FormControl>
														</PopoverTrigger>
														<PopoverContent
															className='w-auto p-0'
															align='start'>
															<Calendar
																onSelect={field.onChange}
																selected={field.value}
																mode='single'
																disabled={(date) =>
																	date < new Date('1900-01-01')
																}
																initialFocus
															/>
														</PopoverContent>
													</Popover>
												</FormItem>
											)}
										/>
									</div>
								</div>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'ministry'}
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
													Ministry
												</FormLabel>
												<Select
													{...{
														onValueChange: field.onChange,
														defaultValue: field.value,
													}}>
													<FormControl>
														<SelectTrigger>
															<SelectValue placeholder='Select a ministry' />
														</SelectTrigger>
													</FormControl>
													<SelectContent>
														{ministries?.map(({ name, id }, index) => (
															<SelectItem value={`${id}`} key={index}>
																{name}
															</SelectItem>
														))}
													</SelectContent>
												</Select>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<FormField
									control={form.control}
									name={'job_type'}
									render={({ field }) => (
										<FormItem>
											<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
												Job type
											</FormLabel>
											<Select
												{...{
													onValueChange: field.onChange,
													defaultValue: field.value,
												}}>
												<FormControl>
													<SelectTrigger>
														<SelectValue placeholder='Select a Job type' />
													</SelectTrigger>
												</FormControl>

												<SelectContent>
													{jobTypes?.map(({ name, id }, index) => (
														<SelectItem {...{ value: `${id}` }} key={index}>
															{name}
														</SelectItem>
													))}
												</SelectContent>
											</Select>
											<FormMessage />
										</FormItem>
									)}
								/>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'location'}
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
													Location
												</FormLabel>
												<FormControl>
													<Input placeholder='Kitui county' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'about_job'}
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
													About job
												</FormLabel>
												<FormControl>
													<Textarea
														placeholder='We are looking for... '
														{...field}
													/>
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className=''>
									<p className='font-bold leading-[24px] text-[16px] text-textTitle'>
										Upload files
									</p>

									<UploadFileCard
										{...{
											handleDeleteRemoteItem,
											jobFiles: job?.files ?? [],
											handleDeleteItem,
											setSelectedFile,
											selectedFiles,
										}}
									/>
								</div>
							</div>

							{/* second section  */}
							<div className='flex-[1] gap-y-4'>
								<FormField
									control={form.control}
									name={'description'}
									render={({ field }) => (
										<FormItem>
											<FormLabel className='text-title-text-color font-bold text-[16px] leading-[24px]'>
												Job Description
											</FormLabel>

											<RichTexEditor
												{...{
													onChangeHandler: (data: string) =>
														form.setValue('description', data),
													defaultValue: field.value,
												}}
											/>
											<FormMessage />
										</FormItem>
									)}
								/>
							</div>
						</div>

						<div className='w-full justify-end flex my-[8px] gap-[48px]'>
							<Button
								role={'button'}
								type={'button'}
								onClick={(e) => router.back()}
								className='flex gap-[8px] bg-white hover:bg-white justify-between px-[16px] py-[10px] rounded-[8px] shadow-btnBoxShadow selection:bg-inherit cursor-pointer border border-border-color'>
								<ArrowLeftIcon
									className='text-gray-body-text'
									width={24}
									height={24}
								/>
								<p className='text-gray-body-text font-semibold text-[14px] leading-[20px]'>
									Back
								</p>
							</Button>

							<Button
								className='rounded-[8px] bg-dev-accent hover:bg-dev-accent text-white border-dev-accent border px-[16px] py-[10px] gap-[8px] shadow-btnBoxShadow w-fit items-center justify-center'
								type='submit'
								{...{
									disabled: loading || job?.status === JobStatusEnum.Closed,
								}}>
								<p className='text-inherit leading-[20px] text-[14px] font-semibold'>
									{loading ? 'Submitting...' : successBtn.text}
								</p>
								{!loading ? (
									<ArrowRightIcon width={20} height={20} color='white' />
								) : null}
							</Button>
						</div>
					</form>
				</Form>
			</div>
			<ToastContainer />
		</div>
	);
};

export const CreateEditUser = ({
	successBtn,
	ministries,
	roles,
}: Pick<TCreateEditJobProps, 'ministries' | 'successBtn'> & {
	roles: Role[];
}) => {
	const [errMessage, setErrMsg] = useState<string | null>(null);
	const [loading, setLoading] = useState(false);
	const router = useRouter();

	const pathname = usePathname();

	const FormSchema = z.object({
		username: z.string().min(2, {
			message: 'Username must be at least 2 characters.',
		}),

		firstname: z.string().min(2, {
			message: 'First name must be at least 2 characters.',
		}),
		lastname: z.string().min(2, {
			message: 'Last name must be at least 2 characters.',
		}),

		phone_number: z.string().min(2, {
			message: 'Phone number must be at least 2 characters.',
		}),

		email: z
			.string()
			.email({ message: 'Email field must contain a valid email' }),

		id_number: z
			.string()
			.min(6, {
				message: 'ID must be at least 6 characters.',
			})
			.max(9, { message: 'ID mus 8 characters or less' }),
		gender: z.string().min(2, {
			message: 'Gender must be at least 2 characters.',
		}),
		position: z.string().min(1, {
			message: 'Position must be field.',
		}),
		ministry: z.string().min(1, {
			message: 'Ministry must be field',
		}),
		accessType: z
			.string()
			.min(1, { message: 'You must provide an access type to a user' }),
	});

	const form = useForm<z.infer<typeof FormSchema>>({
		resolver: zodResolver(FormSchema),
		defaultValues: {
			phone_number: '',
			accessType: '',
			firstname: '',
			id_number: '',
			lastname: '',
			username: '',
			ministry: '',
			position: '',
			gender: '',
			email: '',
		},
	});

	function onSubmit(data: z.infer<typeof FormSchema>) {
		setLoading(true);
		createResourceEndpointData({ url: successBtn.subDetails.url, data })
			.then(({ data: res, err }) => {
				if (err) {
					if (err.status === 400)
						err.details.errors.map(({ path: [field_name], message, name }) =>
							//@ts-ignore
							form.setError(field_name, {
								message: message.replace(field_name, 'This field '),
							})
						);
					else if (err.status === 401)
						router.push(
							`/?${URL_SEARCH_PARAMS.redirect}=${encodeURIComponent(pathname)}`
						);
					else if (err.status === 403) {
						setErrMsg('Permission denied');
						toast.error('Permission denied', {
							position: 'top-right',
						});
					} else {
						toast.error('Something went wrong', {
							position: 'top-right',
						});
						setErrMsg('Something went wrong');
					}
				} else {
					toast.success('User Created', {
						position: 'top-right',
					});
				}
			})
			.finally(() => setLoading(false));
	}

	return (
		<div className=' h-fit m-[10px] flex flex-col gap-[40px] justify-center items-center'>
			<div className='shadow-profileBoxShadow bg-white gap-[4px] p-[28px] rounded-[20px] w[820px]'>
				<div className='flex flex-col gap-[14px] items-center justify-center my-[16px]'>
					<p className='font-bold text-[30px] leading-[36px] tracking-[-.75%] text-title-text-color w-fit'>
						Add a System User
					</p>

					<p className='text-gray-body-text text-[16px] leading-[24px] font-normal'>
						The user below will be able to view job applications and respond to
						them
					</p>
				</div>

				<div className='w-fit mx-auto'>
					<p className='text-red-500'>{errMessage}</p>
				</div>

				<div className='w-full h-fit flex gap-[20px] flex-wrap'>
					<Form {...form}>
						<form
							onSubmit={form.handleSubmit(onSubmit)}
							className='w-full space-y-6'>
							<div className='flex flex-wrap gap-[24px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name='username'
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													Username
												</FormLabel>
												<FormControl>
													<Input placeholder='Name' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name='email'
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													Email
												</FormLabel>
												<FormControl>
													<Input placeholder='Email address' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>
							</div>

							<div className='flex flex-wrap gap-[24px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name='firstname'
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													First name
												</FormLabel>
												<FormControl>
													<Input placeholder='First name' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name='lastname'
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													Lastname
												</FormLabel>
												<FormControl>
													<Input placeholder='Last name' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>
							</div>

							<div className='flex flex-wrap gap-[24px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name='phone_number'
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													Phone
												</FormLabel>
												<FormControl>
													<Input placeholder='Phone' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'id_number'}
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													ID Number
												</FormLabel>
												<FormControl>
													<Input placeholder='' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>
							</div>

							<div className='flex flex-wrap gap-[24px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name='gender'
										render={({ field }) => (
											<FormItem className='space-y-3'>
												<FormLabel>Gender</FormLabel>
												<FormControl>
													<RadioGroup
														onValueChange={field.onChange}
														defaultValue={field.value}
														className='flex flex-col space-y-1'>
														<FormItem className='flex items-center space-x-3 space-y-0'>
															<FormControl>
																<RadioGroupItem value='female' />
															</FormControl>
															<FormLabel className='font-normal'>
																Female
															</FormLabel>
														</FormItem>

														<FormItem className='flex items-center space-x-3 space-y-0'>
															<FormControl>
																<RadioGroupItem value='male' />
															</FormControl>
															<FormLabel className='font-normal'>
																Male
															</FormLabel>
														</FormItem>
													</RadioGroup>
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>
							</div>

							<div className='flex flex-wrap gap-[24px] my-[6px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'ministry'}
										render={({ field }) => (
											<FormItem>
												<FormLabel>Ministry / Department</FormLabel>
												<Select
													onValueChange={field.onChange}
													defaultValue={field.value}>
													<FormControl>
														<SelectTrigger>
															<SelectValue placeholder='Select a ministry' />
														</SelectTrigger>
													</FormControl>
													<SelectContent>
														{ministries?.map(({ name, id }, index) => (
															<SelectItem value={`${id}`} key={index}>
																{name}
															</SelectItem>
														))}
													</SelectContent>
												</Select>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>

								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'position'}
										render={({ field }) => (
											<FormItem>
												<FormLabel className='text-title-text-color'>
													Position
												</FormLabel>
												<FormControl>
													<Input placeholder='Software developer' {...field} />
												</FormControl>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>
							</div>

							<div className='flex flex-wrap gap-[24px] my-[6px]'>
								<div className='flex-[1]'>
									<FormField
										control={form.control}
										name={'accessType'}
										render={({ field }) => (
											<FormItem>
												<FormLabel>Role</FormLabel>
												<Select
													onValueChange={field.onChange}
													defaultValue={field.value}>
													<FormControl>
														<SelectTrigger>
															<SelectValue placeholder='Select a role' />
														</SelectTrigger>
													</FormControl>
													<SelectContent>
														{roles.map(({ name, id }, index) => (
															<SelectItem value={`${id}`} key={index}>
																{name}
															</SelectItem>
														))}
													</SelectContent>
												</Select>
												<FormMessage />
											</FormItem>
										)}
									/>
								</div>
							</div>

							<div className='w-full justify-end flex my-[8px] gap-[32px]'>
								<Button
									role={'button'}
									type={'button'}
									onClick={(e) => router.push(`${BASE_URL}/staff`)}
									className='flex gap-[6px] bg-white hover:bg-white text-black justify-between px-[16px] py-[12px] rounded-[8px] shadow-btnBoxShadow selection:bg-inherit cursor-pointer border border-border-color'>
									Cancel
								</Button>

								<Button
									className='rounded-[8px] bg-dev-accent hover:bg-dev-accent text-white border-dev-accent border px-[40px] py-[12px] gap-[8px] shadow-btnBoxShadow w-fit items-center justify-center'
									type='submit'
									{...{
										disabled: loading,
									}}>
									<p className='text-inherit leading-[20px] text-[14px] font-semibold'>
										{loading ? 'Submitting...' : successBtn.text}
									</p>
								</Button>
							</div>
						</form>
					</Form>
				</div>
			</div>
			<ToastContainer />
		</div>
	);
};
