'use client';

import {
  getPaginationRowModel,
  getFilteredRowModel,
  getCoreRowModel,
  ColumnFiltersState,
  useReactTable,
  VisibilityState,
} from '@tanstack/react-table';
import React, { ChangeEvent, useRef, useState } from 'react';
import { usePathname, useRouter } from 'next/navigation';
import { useForm } from 'react-hook-form';
import { Tooltip } from 'recharts';
import Link from 'next/link';
import { z } from 'zod';

import { zodResolver } from '@hookform/resolvers/zod';

import { Input } from '@/components/ui/input';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';

import { toast, ToastContainer } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

// dev components
import {
  FeatureCategoriesTypes,
  FileSelectorPropsType,
  FilterJobsTypes,
  Media,
  ProfileWrapperPropsTypes,
  TApplicantStatus,
  TComponentsType,
  TDashboardTypes,
  TDropDownProps,
  TDropdownCustomComponent,
  TDropdownCustomComponentRadio,
  TJob,
  TLinegraphReusableComponent,
  TNodes,
  TPaginationComponentsType,
  TReviewStatusProps,
  TTableReusableComponent,
  TUSER,
} from '../utils/types';

import {
  DropdownMenu,
  DropdownMenuCheckboxItem,
  DropdownMenuContent,
  DropdownMenuRadioGroup,
  DropdownMenuRadioItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  ArrowLeftIcon,
  ArrowRightIcon,
  ArrowTopRightIcon,
  CaretDownIcon,
  DownloadIcon,
} from '@radix-ui/react-icons';
import { tickFormatterNumberAbsValue } from '@/utils';
import { LineGraphComponent } from '../reusables/Graph';

import { DataTable } from '../table/Data-table';
import Search from '../reusables/Search';
import { RadioGroup, RadioGroupItem } from '../ui/radio-group';
import { Label } from '../ui/label';
import { Textarea } from '../ui/textarea';
import {
  ApplicantsIcon,
  CalendarIcon,
  CheckMarkIcon,
  ClockIcon,
  CrossMarkIcon,
  DocumentsIcon,
  DropFileIcon,
  EmailIcon,
  HelpIcon,
  LocationIcon,
  SuccessTickIcon,
} from '../icons';
import { FilterIconComponent } from '../reusables/filter';
import AddJobsBtn from '../reusables/Download';
import {
  updateCookie,
  updateResourceEndpointData,
  uploadResourceEndpointData,
} from '../utils/server-functions';
import { Button } from '../ui/button';
import { Checkbox } from '../ui/checkbox';
import { BASE_URL, URL_SEARCH_PARAMS } from '../utils/constants';

export const CustomLinkComponentWithSubLinks = ({
  IconForOpened,
  EndIcon,
  suburls,
  onClick,
  perms,
  name,
  url,
  Icon,
}: TDashboardTypes) => {
  const [opened, setOpened] = useState(false);
  const routePath = usePathname();

  const handleSubrouteTask = () => setOpened((state) => !state);

  return (
    <div
      className={`mx-[2px] my-[2px] p-[12px] selection:bg-inherit align-center rounded-[6px] cursor-pointer hover:bg-dev-accent text-black hover:text-white ${
        routePath.startsWith(url) ? 'bg-foundation-dark-orange  text-white' : ''
      }`}>
      <div
        className='flex justify-between items-center'
        onClick={handleSubrouteTask}>
        <div className='flex gap-[4px]'>
          {Icon ? (
            <Icon
              {...{
                styles: {
                  width: '24px',
                  height: '24px',
                },
                svgElementClassName: routePath.startsWith(url)
                  ? 'stroke-white'
                  : 'dark:stroke-sidebar-bg-light stroke-sidebar-bg',
              }}
            />
          ) : null}

          <p
            {...{
              className: routePath.startsWith(url) ? 'text-white' : '',
            }}>
            {name}
          </p>
        </div>

        {EndIcon && IconForOpened && EndIcon ? (
          opened ? (
            <IconForOpened
              {...{
                styles: {
                  width: '20px',
                  height: '20px',
                },
                svgElementClassName: routePath.startsWith(url)
                  ? 'stroke-white'
                  : 'dark:stroke-sidebar-bg-light stroke-sidebar-bg',
              }}
            />
          ) : (
            <EndIcon
              {...{
                styles: {
                  width: '20px',
                  height: '20px',
                },
                svgElementClassName: routePath.startsWith(url)
                  ? 'stroke-white'
                  : 'dark:stroke-sidebar-bg-light stroke-sidebar-bg',
              }}
            />
          )
        ) : null}
      </div>

      {opened ? (
        <div
          className='flex flex-col h-fit ml-[6px] mt-[16px] gap-[2px]'
          {...{
            onClick,
          }}>
          {suburls?.map(({ url: singleSubUrl, name }, key) => (
            <Link
              className={`hover:bg-[#02020230] px-[6px] py-[6px] rounded-[6px] text-inherit hover:text-white ${
                routePath === `${url}/${singleSubUrl}`
                  ? 'bg-[#02020230] text-white'
                  : ''
              }`}
              href={`${url}/${singleSubUrl}`}
              key={`links-${key}`}>
              {name}
            </Link>
          ))}
        </div>
      ) : null}
    </div>
  );
};

export const CustomLinkComponent = ({
  EndIcon,
  onClick,
  name,
  Icon,
  url,
}: TDashboardTypes) => {
  const routePath = usePathname();

  return (
    <Link
      href={url}
      {...{
        onClick,
      }}>
      <div
        className={`min-h-[46px] h-auto mx-[2px] p-[12px] flex items-center hover:bg-dev-accent hover:text-white gap-[8px] my-[2px] selection:bg-inherit align-center rounded-[6px] cursor-pointer text-body-bg ${
          routePath === url ? 'bg-foundation-dark-orange  text-white' : ''
        }`}>
        {Icon ? (
          <Icon
            {...{
              svgElementClassName:
                routePath === url
                  ? 'stroke-white'
                  : 'stroke-sidebar-bg hover:stroke-white',
              className: 'w-[24px] h-[24px]',
            }}
          />
        ) : null}
        <p
          {...{
            className: `text-inherit ${
              routePath === url ? 'text-white' : 'text-inherit'
            }`,
          }}>
          {name}
        </p>

        {EndIcon ? (
          <div className='w-[20px] h-auto text-body-bg dark:text-white'>
            <CaretDownIcon
              width={20}
              height={20}
            />
          </div>
        ) : null}
      </div>
    </Link>
  );
};

export const Avatar = ({
  classNames = 'w-[32px] h-[32px]',
  name,
}: {
  classNames?: string;
  name: string;
}) => {
  const [fname, lname] = name?.split(' ') || ' ';

  return (
    <p
      className={`${classNames} rounded-[200px] bg-light-purple text-deep-purple flex justify-center items-center content-center`}>
      {fname?.at(0)?.toUpperCase()}
      {lname?.at(0)?.toUpperCase()}
    </p>
  );
};

export const TableComponentHeadings = ({ children }: TNodes) => {
  return (
    <div className='flex flex-col justify-between items-center w-full gap-[24px] flex-wrap px-[24px] py-[10px]'>
      {children}
    </div>
  );
};

export const ShowPagination = ({
  currPageSize,
  onChangeHandler,
}: TPaginationComponentsType) => (
  <div className='flex items-center flex-[1px] gap-[4px]'>
    <p className='font-normal text-[12px] leading-[14.4px]'>Show</p>
    <div className='relative'>
      <input
        {...{
          className:
            'rounded-[8px] h-[36px] w-[54px] outline-none px-[12px] py-[9px] text-[#c3c3c3] bg-sidebar-bg-light dark:bg-small-card-bg text-body dark:text-sidebar-bg-light border border-fwd-darker',
          ...(onChangeHandler
            ? { onChange: onChangeHandler }
            : { readOnly: true }),
          value: currPageSize ?? 10,
          type: 'number',
        }}
      />
      <DropdownMenuTrigger asChild>
        <CaretDownIcon
          className='absolute top-[8px] right-[4px]'
          height={20}
          width={20}
        />
      </DropdownMenuTrigger>
    </div>
    <p className='font-normal text-[12px] leading-[14.4px]'>records</p>
  </div>
);

export const DropDownWrapperCustomComponent = ({
  components,
  children,
}: TNodes & TComponentsType) => {
  return (
    <DropdownMenu>
      {components}
      {children}
    </DropdownMenu>
  );
};
// TDropdownHandlerprops;
export const DropdownCustomComponent = ({
  setFilters,
  width,
  data,
}: TDropdownCustomComponent & Pick<TDropDownProps, 'data' | 'setFilters'>) => (
  <DropdownMenuContent
    className={`dark:text-white w-fit md:w-[60px] bg-sidebar-bg-light text-body dark:bg-small-card-bg ${
      width ? `w-[${width}px]` : 'w-fit'
    }`}>
    {data.map(({ name, checked }, index) => (
      <DropdownMenuCheckboxItem
        {...{
          className: `${width ? `w-[${width}px]` : ''}`,

          onCheckedChange: (checked) =>
            checked
              ? setFilters((prev) => ({
                  ...prev,
                  ministries: [...prev.ministries, name],
                }))
              : setFilters((prev) => ({
                  ...prev,
                  ministries: prev.ministries.filter(
                    (currItem) => currItem !== name
                  ),
                })),
          checked,
        }}
        key={`dropdown-${index}`}
        className='cursor-pointer'>
        {name}
      </DropdownMenuCheckboxItem>
    ))}
  </DropdownMenuContent>
);

export const TooltipComponent = (
  <Tooltip
    {...{
      wrapperClassName:
        'bg-dev-accent rounded-[8px] leading-[15.6px] py-[8px] px-[10px] flex gap-[6px] rounded-[6px] flex-col',
      contentStyle: {
        background: 'var(--accent-color)',
        color: 'var(--accent-light)',
      },
      itemStyle: { padding: '2px', color: 'var(--accent-light)' },
      formatter: tickFormatterNumberAbsValue,
      cursor: false,
    }}
  />
);

export const LineGraphReusableComponent = ({
  graphTitle,
  xAxisLabel,
  graphData,
  yAxisLabel,
  isLoading,
  isError,
  title,
}: TLinegraphReusableComponent) => {
  return (
    <>
      <div className='p-[6px] md:px-[10px] md:pt-[10px] md:pb-[16px] flex items-center justify-between gap-[0px]'>
        <div className='my-[4px]'>
          <h1
            className={`font-bold text-[16px] md:leading-[24px] md:text-[20px] ${'text-left'}`}>
            {title}
          </h1>

          <h1
            className={`font-normal text-[14px] leading-[24px] md:text-[14px] ${'text-left'}`}>
            (This year)
          </h1>
        </div>
      </div>

      {graphTitle ? (
        <div className='py-[8px]'>
          <p className='text-center text-[16px] md:text-[20px] md:leading-[24px] font-bold'>
            {` ${graphTitle}`}
          </p>
        </div>
      ) : null}

      {isLoading ? (
        <div className='h-[360px] w-full'>
          <Loader />
        </div>
      ) : isError ? (
        <div className='h-[400px] md:h-[360px] w-full'>
          <Loader
            {...{
              title: `Error loading data`,
            }}
          />
        </div>
      ) : (
        <div className='h-[360px] w-full'>
          <LineGraphComponent
            {...{
              data: graphData ?? [],
              withAxisLines: {
                x: true,
                y: false,
              },
              xAxisLabel,
              yAxisLabel,
            }}
          />
        </div>
      )}
    </>
  );
};

export const TableReusableComponent = ({
  headerBtn = {
    present: false,
  },
  showPagination = false,
  isSearchAtEnd = false,
  ministryFilterTitle,
  onPageChange,
  filter = true,
  pageCount,
  filterData,
  isLoading,
  pageSize,
  columns,
  isError,
  data,
  title,
}: TTableReusableComponent) => {
  const [columnFilters, setColumnFilters] = useState<ColumnFiltersState>([]);
  const [columnVisibility, setColumnVisibility] = useState<VisibilityState>({
    ['networkBgColor']: false,
  });

  const table = useReactTable({
    getPaginationRowModel: getPaginationRowModel(),
    getFilteredRowModel: getFilteredRowModel(),
    onColumnVisibilityChange: setColumnVisibility,
    onColumnFiltersChange: setColumnFilters,
    getCoreRowModel: getCoreRowModel(),
    columns,
    data,

    state: {
      columnVisibility,
      columnFilters,
    },
  });

  return (
    <React.Fragment>
      <div className='md:col-span-6 rounded-[12px] bg-white shadow-sidebarShadow'>
        <TableComponentHeadings>
          <div className='flex justify-between items-center w-full gap-[24px] flex-wrap px-[24px] py-[10px]'>
            <div className='w-fit'>
              {typeof title === 'string' ? (
                <p className='leading-[28px] text-table-title-color font-bold text-[20px] md:text-[24px] tracking-[-.5%]'>
                  {title}
                </p>
              ) : (
                <React.Fragment>{title}</React.Fragment>
              )}
            </div>

            {!isSearchAtEnd ? (
              <div
                className={`flex items-center justify-center gap-[12px] flex-[2]`}>
                <div className='md:flex items-center justify-center w-full'>
                  <Search
                    {...{
                      title: `Search`,
                      onChangeHandler: (data) =>
                        filterData.setFilters((prev) => ({
                          ...prev,
                          term: data,
                        })),
                      value: filterData.filters.term ?? '',
                    }}
                  />
                </div>
              </div>
            ) : null}

            <div className='flex gap-[12px] items-center'>
              <div className='md:flex items-center justify-end gap-[12px]'>
                {isSearchAtEnd ? (
                  <div
                    className={`flex items-center justify-center gap-[12px] flex-[2]`}>
                    <div className='md:flex items-center justify-center w-full'>
                      <Search
                        {...{
                          title: `Search`,
                          onChangeHandler: (data) =>
                            filterData.setFilters((prev) => ({
                              ...prev,
                              term: data,
                            })),
                          value: filterData.filters.term ?? '',
                        }}
                      />
                    </div>
                  </div>
                ) : null}
                <DropDownWrapperCustomComponent
                  {...{
                    components: <FilterIconComponent />,
                  }}>
                  <DropdownCustomComponent
                    {...{
                      setFilters: filterData.setFilters,
                      data: filterData.data.map(({ name }) => ({
                        checked: filterData.filters.ministries.includes(name),
                        name,
                      })),
                    }}
                  />
                </DropDownWrapperCustomComponent>
              </div>
              {headerBtn.present ? <AddJobsBtn {...headerBtn} /> : null}
            </div>
          </div>

          {filter ? (
            <div className='flex flex-col w-full gap-[24px] px-[24px] py-[10px]'>
              <p className='text-[16px] font-bold leading-[24px] text-center'>
                {ministryFilterTitle}
              </p>

              <div className='flex gap-[16px] overflow-x-auto py-[20px]'>
                {filterData.data.map(({ name }, index) => (
                  <FeaturedJobsCategories
                    key={index}
                    {...{
                      checked: filterData.filters.ministries.find(
                        (value) => value === name
                      )
                        ? true
                        : false,
                      setFilters: filterData.setFilters,
                      name,
                    }}
                  />
                ))}
              </div>
            </div>
          ) : null}
        </TableComponentHeadings>

        {isLoading ? (
          <div className='md:col-span-6 rounded-[12px] bg-white shadow-sidebarShadow min-h-[500px] w-full flex items-center justify-center'>
            <Loader />
          </div>
        ) : isError ? (
          <div className='md:col-span-6 rounded-[12px] bg-white shadow-sidebarShadow min-h-[500px] w-full flex items-center justify-center'>
            <Loader
              {...{
                title: 'Loading data...',
              }}
            />
          </div>
        ) : (
          <DataTable
            {...{
              pagination: showPagination,
              pageCount: pageCount,
              pageSize: pageSize,
              onPageChange,
              table,
            }}
          />
        )}
      </div>
    </React.Fragment>
  );
};

export const Loader = ({ title = 'Fetching data...' }: { title?: string }) => (
  <div className='w-full h-full flex justify-center items-center'>{title}</div>
);

export const UploadedDocument = ({ file }: { file: Media }) => {
  return (
    <div className='w-full flex justify-between items-center gap-[4px] border-b-[1px] py-[6px] my-[6px]'>
      <div className='flex-[.4]'>
        <DocumentsIcon
          {...{
            svgElementClassName: 'stroke-dev-accent',
            applyToSvgEl: false,
            styles: {
              width: '24px',
              height: '24px',
            },
          }}
        />
      </div>

      <div className='leading-[24px] text-[16px] font-normal text-gray-body-text flex-[10]'>
        {file.name}
      </div>
      <div className='flex-[5] flex gap-[6px] items-center justify-end'>
        <Link
          download={file.url}
          target='_blank'
          href={file.url}
          className='flex gap-[8px] border-purple-800 border px-[14px] py-[8px] rounded-[8px] cursor-pointer selection:bg-inherit'>
          <DownloadIcon className='text-purple-800 leading-[20px] text-[14px] font-semibold' />
          <p className='text-purple-800 leading-[20px] text-[14px] font-semibold'>
            Download
          </p>
        </Link>
      </div>
    </div>
  );
};

export const PersonalDetails = ({
  user,
  files,
  onClick,
}: {
  onClick: () => void;
  user: TUSER;
  files?: any[];
}) => {
  const FormSchema = z.object({
    countyResidence: z.string(),
    nationalId: z.string(),
    subCounty: z.string(),
    location: z.string(),
    gender: z.string(),
    phone: z.string(),
    name: z.string(),
    email: z.string(),
  });

  const form = useForm<z.infer<typeof FormSchema>>({
    resolver: zodResolver(FormSchema),
    defaultValues: {
      name: user?.firstname ? `${user?.firstname} ${user?.lastname}` : '',
      countyResidence: user?.county ?? '',
      nationalId: user?.id_number ?? '',
      subCounty: user?.sub_county ?? '',
      phone: user?.phone_number ?? '',
      location: user?.location ?? '',
      gender: user?.gender ?? '',
      email: user?.email ?? '',
    },
  });

  return (
    <div className='w-full h-fit flex gap-[40px] md:p-[28px] flex-wrap'>
      <Form {...form}>
        <form className='w-full space-y-6'>
          <div className='flex flex-wrap gap-[12px]'>
            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name='name'
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      Name
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...{ disabled: true }}
                        placeholder='Name'
                        {...field}
                        className=''
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name='email'
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      Email
                    </FormLabel>
                    <FormControl>
                      <Input
                        placeholder='Email address'
                        {...field}
                        {...{ disabled: true }}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </div>

          <div className='flex flex-wrap gap-[24px]'>
            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name='phone'
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      Phone
                    </FormLabel>
                    <FormControl>
                      <Input
                        placeholder='Phone'
                        {...field}
                        {...{ disabled: true }}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name={'nationalId'}
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      ID Number
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...{ disabled: true }}
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </div>

          <div className='flex flex-wrap gap-[24px]'>
            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name='gender'
                render={({ field }) => (
                  <FormItem
                    className='space-y-3'
                    {...{ disabled: true }}>
                    <FormLabel>Gender</FormLabel>
                    <FormControl>
                      <RadioGroup
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                        {...{ disabled: true }}
                        className='flex flex-col space-y-1'>
                        <FormItem className='flex items-center space-x-3 space-y-0'>
                          <FormControl>
                            <RadioGroupItem value='female' />
                          </FormControl>
                          <FormLabel className='font-normal'>Female</FormLabel>
                        </FormItem>

                        <FormItem className='flex items-center space-x-3 space-y-0'>
                          <FormControl>
                            <RadioGroupItem value='male' />
                          </FormControl>
                          <FormLabel className='font-normal'>Male</FormLabel>
                        </FormItem>
                      </RadioGroup>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </div>

          <div className='flex flex-wrap gap-[24px] my-[6px]'>
            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name={'countyResidence'}
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      County of residence
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        {...{ disabled: true }}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </div>

          <div className='flex flex-wrap gap-[24px]'>
            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name={'subCounty'}
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      Sub county
                    </FormLabel>
                    <FormControl>
                      <Input
                        placeholder='Phone'
                        {...{ disabled: true }}
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <div className='flex-[1]'>
              <FormField
                control={form.control}
                name={'location'}
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className='text-title-text-color'>
                      Location
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...{ disabled: true }}
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </div>
        </form>
      </Form>

      <div className='w-full my-[6px] mt-[10px]'>
        <p className='font-bold leading-[24px] text-title-text-color'>
          Resume / CV
        </p>
        <div className='w-full'>
          {files && files?.length > 0 ? (
            files?.map((currFile) => (
              <UploadedDocument {...{ file: currFile }} />
            ))
          ) : (
            <p>No files where attached to this application</p>
          )}
        </div>
      </div>

      <div className='w-full justify-end flex my-[8px]'>
        <button
          className='bg-dev-accent w-fit h-fit flex gap-[10px] rounded-[8px] px-[16px] py-[10px] border items-center justify-center selection:bg-inherit'
          onClick={onClick}
          role='button'>
          <p className='text-white'>Review Application Status</p>
          <ArrowRightIcon
            width={24}
            height={24}
            color='white'
          />
        </button>
      </div>
    </div>
  );
};

export const ReviewStatus = ({
  canGoToNext = true,
  selectedStatus,
  savedComment,
  handlePrevious,
  handleChange,
  handleNext,
  comment,
  status,
  errMsg,
  loading,
}: TReviewStatusProps) => {
  return (
    <div className='w-full h-fit'>
      <div className='w-full mt-[16px]'>
        {errMsg ? (
          <p className='text-red-500 text-center'>
            {errMsg}, review couldn't be posted
          </p>
        ) : null}
        {status === TApplicantStatus.Awarded ? (
          <p className='text-[14px] font-bold text-center px-[20px] py-[6px] my-[20px] text-job-awarded-text-color bg-job-awarded-bg-color rounded-[20px] w-fit mx-auto'>
            This job has been awarded. No further action required.
          </p>
        ) : status === TApplicantStatus.Rejected ? (
          <p className='text-[14px] font-bold text-center px-[20px] py-[6px] my-[20px] text-job-closed-text-color bg-job-closed-bg-color rounded-[20px] w-fit mx-auto'>
            This applicant has been rejected. No further action required.
          </p>
        ) : null}

        <Label className='text-[16px] font-bold leading-[24px] text-title-text-color'>
          Application status
        </Label>
        <RadioGroup
          className='my-[10px]'
          {...{
            ...(status === TApplicantStatus.Awarded ||
            status === TApplicantStatus.Rejected
              ? { disabled: true }
              : {}),
            defaultValue:
              status === TApplicantStatus.UnderReview
                ? TApplicantStatus.UnderReview
                : status === TApplicantStatus.Shortlisted
                ? TApplicantStatus.Shortlisted
                : status === TApplicantStatus.Rejected
                ? TApplicantStatus.Rejected
                : TApplicantStatus.Awarded
                ? TApplicantStatus.Awarded
                : status === TApplicantStatus.New
                ? TApplicantStatus.New
                : '',
          }}
          onValueChange={(status: TApplicantStatus) =>
            handleChange((prev) => ({ ...prev, status }))
          }>
          {status === TApplicantStatus.New ? (
            <>
              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.UnderReview}
                  id='r1'
                />
                <Label htmlFor='r1'>Under review</Label>
              </div>
              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Rejected}
                  id='r2'
                />
                <Label htmlFor='r2'>Reject</Label>
              </div>

              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Shortlisted}
                  id='r3'
                />
                <Label htmlFor='r3'>Shortlist</Label>
              </div>
            </>
          ) : status === TApplicantStatus.UnderReview ? (
            <>
              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Rejected}
                  id='r1'
                />
                <Label htmlFor='r1'>Reject</Label>
              </div>

              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Shortlisted}
                  id='r2'
                />
                <Label htmlFor='r2'>Shortlist</Label>
              </div>
            </>
          ) : status === TApplicantStatus.Shortlisted ? (
            <>
              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Rejected}
                  id='r1'
                />
                <Label htmlFor='r1'>Reject</Label>
              </div>

              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Awarded}
                  id='r2'
                />
                <Label htmlFor='r2'>Award</Label>
              </div>
            </>
          ) : status === TApplicantStatus.Awarded ? (
            <>
              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Shortlisted}
                  id='r2'
                />
                <Label htmlFor='r2'>Shortlist</Label>
              </div>

              <div className='flex items-center space-x-2 gap-[6px]'>
                <RadioGroupItem
                  value={TApplicantStatus.Awarded}
                  id='r2'
                />
                <Label htmlFor='r2'>Award</Label>
              </div>
            </>
          ) : savedComment ? (
            <div className='w-full'>
              <p className=''>Previous comments</p>
              <p className='text-gray-500'>{savedComment}</p>
            </div>
          ) : null}
        </RadioGroup>
      </div>

      <div className='my-[6px]'>
        <Label className='font-bold leading-[24px] text-[16px] text-title-text-color'>
          Comment
        </Label>

        <Textarea
          placeholder='Type your comment here.'
          {...{
            ...(status === TApplicantStatus.Awarded ||
            status === TApplicantStatus.Rejected
              ? { disabled: true }
              : {}),
            value: comment,
          }}
          onChange={(e) =>
            handleChange((prev) => ({
              ...prev,
              comment: e.target.value,
            }))
          }
        />
        <p className='leading-[24px] text-[12px] font-semibold text-gray-400'>
          Share your thoughts on the candidate here
        </p>
      </div>

      <div className='w-full flex justify-end gap-[12px]'>
        <button
          className='border-border-color w-fit h-fit flex gap-[10px] rounded-[8px] px-[16px] py-[10px] border items-center justify-center selection:bg-inherit'
          onClick={handlePrevious}
          role='button'
          {...{
            ...(loading ? { disabled: true } : {}),
          }}>
          <ArrowLeftIcon
            width={24}
            height={24}
          />
          <p className='text-black w-fit'>Back</p>
        </button>

        {canGoToNext ? (
          <button
            className={`bg-dev-accent w-fit h-fit flex gap-[10px] rounded-[8px] px-[16px] py-[10px] border items-center justify-center selection:bg-inherit`}
            onClick={handleNext}
            role='button'
            {...{
              ...(loading || !selectedStatus ? { disabled: true } : {}),
              ...(TApplicantStatus.Awarded
                ? {
                    title: 'This candidate or job has been concluded.',
                  }
                : {}),
            }}>
            <p className='text-white w-fit'>
              {loading ? 'Saving...' : 'Save Assesment'}
            </p>
            <ArrowRightIcon
              width={24}
              height={24}
              color='white'
            />
          </button>
        ) : null}
      </div>
    </div>
  );
};

export const CompletedAssesment = ({ user }: { user: TUSER }) => {
  const router = useRouter();
  return (
    <div className='w-full h-fit'>
      <div className='w-full flex justify-center'>
        <SuccessTickIcon
          {...{
            svgElementClassName: 'stroke-tick-color stroke-[10px]',
            applyToSvgEl: false,
            styles: {
              width: '250px',
              height: '250px',
            },
          }}
        />
      </div>

      <div className='flex items-center flex-col'>
        <p className='font-bold text-[30px] leading-[36px] text-center'>
          Your assessment has been saved
        </p>
        <div className='text-gray-body-text font-normal leading-[24px] text-[16px] text-center my-[6px]'>
          Thank you for reviewing{' '}
          <p className='inline font-bold leading-[24px] text-[16px] text-center text-inherit'>
            {user.username}’s
          </p>{' '}
          application. You can also send the applicant an email if they are
          shortlisted.
        </div>
      </div>

      <div className='w-full flex justify-center gap-[20px] my-[8px]'>
        <div
          className='border-border-color w-fit h-fit flex gap-[10px] rounded-[8px] px-[16px] py-[10px] border items-center justify-center selection:bg-inherit'
          role='button'
          onClick={(e) => router.back()}>
          <ArrowLeftIcon
            width={24}
            height={24}
          />
          <p className='text-black w-fit'>Previous page</p>
        </div>

        <div
          className='bg-dev-accent w-fit h-fit flex gap-[10px] rounded-[8px] px-[16px] py-[10px] border items-center justify-center selection:bg-inherit'
          role='button'>
          <EmailIcon
            {...{
              svgElementClassName: 'stroke-white',
              applyToSvgEl: false,
              styles: {
                width: '24px',
                height: '24px',
              },
            }}
          />
          <p className='text-white w-fit'>Email Candidate</p>
        </div>
      </div>
    </div>
  );
};

export const JobType = ({
  className = 'border-brown-border text-brown-text px-[12px] py-[4px] rounded-[40px]',
  name = 'Attachment',
}: {
  className?: string;
  name?: string;
}) => {
  return (
    <p
      className={`border  h-fit font-bold  selection:bg-inherit rounded-[40px] px-[12px] py-[4px] leading-[16.37px] text-[12px] ${className}`}>
      {name}
    </p>
  );
};

export const LocationLabel = ({ name }: { name?: string }) => (
  <div className='flex gap-[4px] items-center justify-center'>
    <LocationIcon
      {...{
        styles: {
          width: '20px',
          height: '20px',
        },
        svgElementClassName: 'stroke-gray-body-text',
      }}
    />
    <p>{name}</p>
  </div>
);

export const TimeLimitLabel = ({ time = '2 days left' }: { time?: string }) => (
  <div className='flex gap-[4px] items-center justify-center'>
    <ClockIcon
      {...{
        styles: {
          width: '20px',
          height: '20px',
        },
        svgElementClassName: 'stroke-gray-body-text',
      }}
    />
    <p className='text-time-color'>{time}</p>
  </div>
);

export const JobMinistryTag = ({
  textClassName = 'text-dev-accent',
  ministry_name = 'Youth & Culture',
  className = 'bg-tag-color',
  dotClass = 'bg-dev-accent',
}: {
  textClassName?: string;
  ministry_name: string;
  className?: string;
  dotClass?: string;
}) => (
  <div
    className={`rounded-[40px] gap-[4px] px-[12px] py-[4px] flex  min-h[24px] items-center ${className}`}>
    <p className={`w-[6px] h-[6px] rounded-full ${dotClass}`} />
    <p className={`font-bold leading-[16.37px] text-[12px] ${textClassName}`}>
      {ministry_name}
    </p>
  </div>
);

export const ApplicantCountIcon = ({
  label = 'Applications',
  count = 0,
  children,
}: {
  count?: number;
  label?: string;
} & TNodes) => {
  return (
    <div className='flex gap-[6px] items-center justify-center'>
      <ApplicantsIcon
        {...{
          styles: {
            width: '28px',
            height: '28px',
          },
          svgElementClassName: 'stroke-gray-body-text',
        }}
      />

      <p className='text-gray-body-text text-[14px] leading-[24px] font-normal'>
        {count} {label}
      </p>

      {children ? children : null}
    </div>
  );
};

export const PageForPagination = ({
  active = false,
  content,
  handler,
}: {
  handler: (data: any) => void;
  content: string;
  active?: boolean;
}) => (
  <div
    className={`rounded-[8px] selection:bg-inherit flex items-center justify-center leading-[24px] hover:bg-[#F9F5FF] text-[14px] font-medium text-center cursor-pointer w-[40px] h-[40px] ${
      active ? 'bg-[#F9F5FF] text-[#7F56D9]' : 'text-main-orange'
    }`}
    onClick={handler}>
    {content}
  </div>
);

export const FileListDivider = () => (
  <div className='bg-dividerColor h-[1px]'></div>
);

export const SelectedFileSingleList = ({
  handleDeleteRemoteItem,
  handleDeleteItem,
  remoteFile,
  file,
}: {
  remoteFile?: Media;
  file?: File;
} & Pick<
  FileSelectorPropsType,
  'handleDeleteItem' | 'handleDeleteRemoteItem'
>) => {
  return (
    <div className='flex flex-col gap-[8px]'>
      <div className='flex gap-[4x] justify-between items-center'>
        <CheckMarkIcon
          {...{
            svgElementClassName: 'stroke-checkMarkColor',
            className: 'w-[24px] h-[24px]',
            applyToSvgEl: true,
          }}
        />
        <div className='flex-[1] px-[6px]'>
          <p className='font-normal leading-[25.78px] text-[16px] text-bodyText overflow-x-hidden'>
            {remoteFile ? remoteFile.name : file?.name}
          </p>
        </div>

        <div
          onClick={(e) =>
            remoteFile
              ? handleDeleteRemoteItem
                ? handleDeleteRemoteItem({ id: remoteFile.id })
                : null
              : handleDeleteItem
              ? handleDeleteItem(file!)
              : null
          }
          className='w-fit'
          title='Delete file'>
          <CrossMarkIcon
            {...{
              svgElementClassName: 'stroke-crossMarkColor hover:stroke-red-700',
              className: 'w-[24px] h-[24px]',
              applyToSvgEl: true,
            }}
          />
        </div>
      </div>
    </div>
  );
};

export const UploadFileCard = ({
  handleDeleteRemoteItem,
  handleDeleteItem,
  setSelectedFile,
  selectedFiles,
  jobFiles = [],
}: FileSelectorPropsType) => {
  const inputFieldRef = useRef<HTMLInputElement | null>(null);
  const triggerFileSelectorDialog = (e: any) => inputFieldRef.current?.click();

  const handleSelectedFile = (e: ChangeEvent<HTMLInputElement>) => {
    setSelectedFile((prev) =>
      e.currentTarget?.files
        ? [...prev, ...Array.from(e.currentTarget?.files)]
        : prev
    );
    if (inputFieldRef.current) inputFieldRef.current.value = '';
  };

  return (
    <div className='border rounded-[8px] px-[18px] py-[20px] border-checkboxColor'>
      <div
        className='w-full'
        title='Select file or drag and drop here'>
        <div
          className='flex gap-[60px] cursor-pointer selection:bg-inherit'
          onClick={triggerFileSelectorDialog}
          onDrag={triggerFileSelectorDialog}>
          <div className='px-[14px] py-[8px] rounded-[4px] bg-chooseFileBgColor'>
            <p className='leading-[19.36px] text-[16px] font-medium text-checkMarkColor'>
              Choose file
            </p>
          </div>
          <div className='flex gap-[4px]'>
            <DropFileIcon
              {...{
                svgElementClassName: 'stroke-chooseFileIconBgColor',
                applyToSvgEl: false,
                styles: {
                  width: '24px',
                  height: '24px',
                },
              }}
            />
            <p className='text-chooseFileTextColor leading-[25.78px] text-[16px] font-normal'>
              Drop your documents here
            </p>
          </div>
        </div>

        <input
          {...{
            onChange: handleSelectedFile,
            accept: '.pdf, .doc, .docx',
            multiple: true,
            id: 'picture',
            type: 'file',
          }}
          ref={inputFieldRef}
          className='hidden'
        />
      </div>

      <div className='flex flex-col gap-[8px] my-[10px]'>
        {selectedFiles.map((file, index) => (
          <React.Fragment key={index}>
            <SelectedFileSingleList {...{ file, handleDeleteItem }} />
            {index !== selectedFiles.length - 1 ? <FileListDivider /> : null}
          </React.Fragment>
        ))}

        {jobFiles.map((remoteFile, index) => (
          <React.Fragment key={index}>
            <SelectedFileSingleList
              {...{ remoteFile, handleDeleteRemoteItem }}
            />
            {index !== selectedFiles.length - 1 ? <FileListDivider /> : null}
          </React.Fragment>
        ))}
      </div>
    </div>
  );
};

export const HumbuggerComponent = ({
  onChangeHandler: clickHandler,
  opened,
}: {
  onChangeHandler: (data: any) => void;
  opened: boolean;
}) => (
  <div
    className='w-[28px]'
    onClick={clickHandler}>
    <p
      className={`h-[3px] bg-body dark:bg-white w-full my-[3px] transition-transform duration-500 ${
        opened ? 'rotate-[26deg] origin-left' : ''
      }`}></p>

    <p
      className={`h-[3px] bg-body dark:bg-white w-full my-[3px] transition-all duration-200 ${
        opened ? 'invisible' : ''
      }`}></p>
    <p
      className={`h-[3px] bg-body dark:bg-white w-full my-[3px]  transition-transform duration-500 ${
        opened ? '-rotate-[26deg] origin-left' : ''
      }`}></p>
  </div>
);

export const DropdownCustomComponentRadio = ({
  onChangeHandler,
  currPageSize,
  width,
  data,
}: TDropdownCustomComponentRadio) => (
  <DropdownMenuContent
    className={`dark:text-white w-fit md:w-[60px] bg-sidebar-bg-light text-body dark:bg-small-card-bg ${
      width ? `w-[${width}px]` : 'w-fit'
    }`}>
    <DropdownMenuRadioGroup
      {...{
        className: `${width ? `w-[${width}px]` : ''}`,

        onValueChange: (value) => {
          const foundTargetObject = data.find(({ label }) => label === value);
          foundTargetObject?.onChangeHandler
            ? foundTargetObject?.onChangeHandler({
                label: foundTargetObject.label,
                id: parseInt(`${foundTargetObject.id}`),
              })
            : onChangeHandler({ ...foundTargetObject, id: currPageSize });
        },
        value: `${currPageSize!}`,
      }}>
      {data.map(({ label }, index) => (
        <DropdownMenuRadioItem
          key={`dropdown-${index}`}
          className='cursor-pointer'
          {...{
            value: label,
          }}>
          {label}
        </DropdownMenuRadioItem>
      ))}
    </DropdownMenuRadioGroup>
  </DropdownMenuContent>
);

export const FeaturedJobsCategories = ({
  setFilters,
  checked,
  name,
}: FeatureCategoriesTypes & Pick<FilterJobsTypes, 'checked'>) => {
  return (
    <div
      role='button'
      className={`flex-grow-0 flex-shrink-0 flex-basis-auto flex justify-between max-w-[300px] px-[18px] selection:bg-inherit border py-[10px] ${
        checked
          ? 'bg-foundation-dark-orange border-foundation-dark-orange text-white'
          : 'bg-white border-[#9CA3AF] text-navy'
      } md:border md:border-purple-300 md:rounded-[40px] justify-between items-center gap-[4px]`}
      {...{
        onClick: () =>
          checked
            ? setFilters((prev) => ({
                ...prev,
                ministries: prev.ministries.filter(
                  (currItem) => currItem !== name
                ),
              }))
            : setFilters((prev) => ({
                ...prev,
                ministries: [...prev.ministries, name],
              })),
        checked,
      }}>
      <div
        className={`rounded-full  bg-white w-[8px] h-[8px] ml-2 ${
          checked ? '' : 'hidden'
        }`}></div>
      <p
        className={`text-bodyBg font-bold text-[12px] leading-[16.37px] ${
          checked ? 'text-bodyBg' : 'text-bodyText'
        }`}>
        {name}
      </p>
    </div>
  );
};

export const ProfileWrapper = ({ data, refetch }: ProfileWrapperPropsTypes) => {
  const [errMessage, setErrMsg] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);
  const pathname = usePathname();
  const router = useRouter();

  const FormSchema = z.object({
    username: z.string().min(2, {
      message: 'Username must be at least 2 characters.',
    }),

    location: z.string().min(2, {
      message: 'Location must be at least 2 characters.',
    }),

    firstname: z.string().min(2, {
      message: 'First name must be at least 2 characters.',
    }),

    lastname: z.string().min(2, {
      message: 'Last name must be at least 2 characters.',
    }),

    phone_number: z.string().min(2, {
      message: 'Phone number must be at least 2 characters.',
    }),

    id_number: z
      .string()
      .min(6, {
        message: 'ID must be at least 6 characters.',
      })
      .max(9, { message: 'ID mus 8 characters or less' }),

    gender: z.string().min(2, {
      message: 'Gender must be at least 2 characters.',
    }),
  });

  const form = useForm<z.infer<typeof FormSchema>>({
    resolver: zodResolver(FormSchema),
    defaultValues: {
      phone_number: data?.data.phone_number ?? '',
      id_number: data?.data.id_number ?? '',
      firstname: data?.data.firstname ?? '',
      username: data?.data.username ?? '',
      location: data?.data?.location ?? '',
      lastname: data?.data.lastname ?? '',
      gender: data?.data.gender ?? '',
    },
  });

  function onSubmit(data: z.infer<typeof FormSchema>) {
    const frmData = new FormData();

    frmData.append('data', `${JSON.stringify(data)}`);
    setLoading(true);

    uploadResourceEndpointData({
      data: frmData,
      url: `auth/users`,
      method: 'PUT',
    })
      .then(({ data: responseData, err }) => {
        if (err) {
          if (err.status === 400)
            err.details.errors &&
              err.details.errors.map(({ path: [field_name], message }) =>
                //@ts-ignore
                form.setError(field_name, {
                  message: message.replace(field_name, 'This field '),
                })
              );
          else if (err.status === 401)
            router.push(
              `/?${URL_SEARCH_PARAMS.redirect}=${encodeURIComponent(pathname)}`
            );
          else if (err.status === 403) {
            setErrMsg('Permission denied');
            toast.error('Permission denied', {
              position: 'top-right',
            });
          } else {
            setErrMsg('Something went wrong');
            toast.error('Something went wrong', {
              position: 'top-right',
            });
          }
        } else {
          updateCookie({ user: responseData, updatedData: data });
          refetch();
          toast.success('Profile Updated', {
            position: 'top-right',
          });
        }
      })
      .finally(() => setLoading(false));
  }

  return (
    <div className=' h-full m-[10px] flex flex-col gap-[40px] px-[16px]'>
      <div className='flex gap-[24px] items-center w-fit'>
        <Avatar
          {...{
            classNames: 'w-[130px] h-[130px] font-extrabold',
            name: data?.data?.username ?? '',
          }}
        />
        <div className='flex gap-[10px] flex-col'>
          <p className='text-center tracking-[-.5%] leading-[30.01px] text-[22px] font-bold text-title-text-color'>
            {data?.data?.username ?? ''}
          </p>
          <p className='text-gray-body-text text-center leading-[24px] text-[16px] font-normal'>
            {data?.data?.email ?? ''}
          </p>
        </div>
      </div>
      <div className='shadow-profileBoxShadow bg-white gap-[4px] p-[28px] rounded-[20px]'>
        <p className='text-title-text-color tracking-[-.5%] leading-[28px] text-[24px] font-bold'>
          Profile details
        </p>

        <div className='w-fit mx-auto'>
          <p className='text-red-500'>{errMessage}</p>
        </div>

        <div className='w-full h-fit flex gap-[20px] flex-wrap'>
          <Form {...form}>
            <form
              onSubmit={form.handleSubmit(onSubmit)}
              className='w-full space-y-6'>
              <div className='flex flex-wrap gap-[24px]'>
                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name='username'
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className='text-title-text-color'>
                          Username
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder='Name'
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name='location'
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className='text-title-text-color'>
                          Location
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder='Kibatu'
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </div>

              <div className='flex flex-wrap gap-[24px]'>
                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name='firstname'
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className='text-title-text-color'>
                          First name
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder='First name'
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name='lastname'
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className='text-title-text-color'>
                          Lastname
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder='Last name'
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </div>

              <div className='flex flex-wrap gap-[24px]'>
                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name='phone_number'
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className='text-title-text-color'>
                          Phone
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder='Phone'
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name={'id_number'}
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className='text-title-text-color'>
                          ID Number
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder=''
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </div>

              <div className='flex flex-wrap gap-[24px]'>
                <div className='flex-[1]'>
                  <FormField
                    control={form.control}
                    name='gender'
                    render={({ field }) => (
                      <FormItem className='space-y-3'>
                        <FormLabel>Gender</FormLabel>
                        <FormControl>
                          <RadioGroup
                            onValueChange={field.onChange}
                            defaultValue={field.value}
                            className='flex flex-col space-y-1'>
                            <FormItem className='flex items-center space-x-3 space-y-0'>
                              <FormControl>
                                <RadioGroupItem value='female' />
                              </FormControl>
                              <FormLabel className='font-normal'>
                                Female
                              </FormLabel>
                            </FormItem>

                            <FormItem className='flex items-center space-x-3 space-y-0'>
                              <FormControl>
                                <RadioGroupItem value='male' />
                              </FormControl>
                              <FormLabel className='font-normal'>
                                Male
                              </FormLabel>
                            </FormItem>
                          </RadioGroup>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </div>

              <div className='flex flex-wrap gap-[24px] my-[6px]'>
                <div className='flex-[1] flex-col gap-[10px]'>
                  <FormLabel className='text-title-text-color'>
                    Department
                  </FormLabel>
                  <Input
                    {...{
                      value: data?.data.ministry?.name,
                      readOnly: true,
                      disabled: true,
                    }}
                  />
                </div>

                <div className='flex-[1] flex-col gap-[10px]'>
                  <FormLabel className='text-title-text-color'>
                    Position
                  </FormLabel>
                  <Input
                    {...{
                      value: data?.data.staff_role,
                      readOnly: true,
                      disabled: true,
                    }}
                  />
                </div>
              </div>

              <div className='w-full justify-center flex my-[8px] gap-[32px]'>
                <Button
                  role={'button'}
                  type={'button'}
                  onClick={(e) => router.back()}
                  {...{
                    disabled: loading,
                  }}
                  className='flex gap-[6px] bg-white hover:bg-white text-black justify-between px-[16px] py-[12px] rounded-[8px] shadow-btnBoxShadow selection:bg-inherit cursor-pointer border border-border-color'>
                  Back
                </Button>

                <Button
                  className='rounded-[8px] bg-dev-accent hover:bg-dev-accent text-white border-dev-accent border px-[40px] py-[12px] gap-[8px] shadow-btnBoxShadow w-fit items-center justify-center'
                  type='submit'
                  {...{
                    disabled: loading,
                  }}>
                  Update details
                </Button>
              </div>
            </form>
          </Form>
        </div>
      </div>

      <ToastContainer />
    </div>
  );
};

export const MarkAsSeen = () => ({
  accessorKey: 'seen',
  header: () => {
    return (
      <div className='w-full flex gap-[2px] items-center'>
        <p className='text-gray-body-text font-medium text-[12px] leading-[18px]'>
          Mark as seen
        </p>
        <HelpIcon
          {...{
            styles: { height: '16px', width: '16px' },
            svgElementClassName: 'stroke-dots-color',
          }}
        />
      </div>
    );
  },
  cell: ({ row }: any) => {
    const { seen: initialSeen, id } = row.original;
    const [isChecked, setIsChecked] = useState(initialSeen);
    const router = useRouter();

    const [loading, setLoading] = useState(false);

    const handleMarkChecked = (seen: boolean) => {
      setLoading(true);
      updateResourceEndpointData({
        url: `applications/${id}`,
        data: { seen },
        id: `${id}`,
      })
        .then(({ err }) => {
          if (err) {
            if (err.status === 400) {
              toast.error('Could not update the status', {
                position: 'top-right',
              });
            } else if (err.status === 401) {
              toast.error('Your session has expired', {
                position: 'top-right',
              });
              router.push('/');
            } else if (err.status === 403)
              toast.error('Permission denied', {
                position: 'top-right',
              });
            else {
              toast.error('Something went wrong', {
                position: 'top-right',
              });
            }
            setIsChecked(!seen);
          } else {
            toast.success('Succesffully updated the candidate status', {
              position: 'top-right',
            });
            setIsChecked(seen);
          }
        })
        .finally(() => setLoading(false));
    };

    return (
      <div className='h-fit flex gap-[12px]  items-center px-[24px] py-[16px]'>
        <Checkbox
          className='border border-border-color data-[state=checked]:border-[2px] data-[state=checked]:border-dev-accent bg-white data-[state=checked]:bg-hover-bg-color-btn'
          {...{
            onCheckedChange: handleMarkChecked,
            checked: isChecked,
            disabled: loading,
          }}
        />
        <ToastContainer />
      </div>
    );
  },
});

export const JobDetails = ({
  viewJob = true,
  label,
  job,
}: {
  job?: TJob;
  label?: string;
  viewJob?: boolean;
}) => {
  const router = useRouter();
  return (
    <div className='flex flex-col justify-center items-center gap-[6px]'>
      <p className='font-medium text-[20px] leading-[28px]'>Job Applied for:</p>

      <div className='flex gap-[6px] items-center'>
        <p className='font-bold text-[30px] leading-[36px]'>{job?.title}</p>
        <div className='w-fit'>
          <JobMinistryTag
            {...{
              ministry_name: job?.ministry.name ?? '',
            }}
          />
        </div>
      </div>

      <div className='flex gap-[20px] my-[4px] justify-center items-center'>
        <div className='w-fit'>
          <JobType {...{ name: job?.job_type.name }} />
        </div>

        <div className='w-fit'>
          <LocationLabel {...{ name: job?.location }} />
        </div>

        <div className='w-fit flex gap-[2px] items-center'>
          <CalendarIcon
            {...{
              styles: {
                width: '24px',
                height: '24px',
              },
              svgElementClassName: 'stroke-black',
            }}
          />
          <p>
            {new Date().toLocaleString('en', {
              month: 'short',
              year: 'numeric',
              day: '2-digit',
            })}
          </p>
        </div>
      </div>

      <div
        className={`flex ${
          viewJob ? 'gap-[4px]' : 'gap-[20px]'
        } items-center my-[10px]`}>
        <div className='w-fit'>
          <ApplicantCountIcon
            {...{
              count: job?.applicationCount,
              label,
              ...(!viewJob
                ? {
                    children: (
                      <div className='w-fit'>
                        <HelpIcon
                          {...{
                            svgElementClassName: 'stroke-dots-color',
                            styles: { height: '16px', width: '16px' },
                          }}
                        />
                      </div>
                    ),
                  }
                : {}),
            }}
          />
        </div>

        {viewJob ? (
          <div
            className='flex gap-[2px] rounded-[8px] border border-purple-400 justify-between text-purple-800 px-[14px] py-[8px] items-centerjustify-between w-fit shadow-btnBoxShadow selection:bg-inherit cursor-pointer'
            onClick={(e) => router.push(`${BASE_URL}/jobs/${job?.id}`)}>
            <p className='font-semibold text-[14px] leading-[20px]'>View Job</p>
            <ArrowTopRightIcon
              {...{
                width: '20',
                height: '20',
              }}
            />
          </div>
        ) : null}
      </div>
    </div>
  );
};
