'use client';
import { TCardProps, TCommentType, TMinistry } from '../utils/types';
import { ArrowUpIcon, DeleteIcon, MessageIcon } from '../icons';
import { ArrowRightIcon } from '@radix-ui/react-icons';
import { formatDistanceStrict } from 'date-fns';

import {
	ApplicantCountIcon,
	JobMinistryTag,
	TimeLimitLabel,
	LocationLabel,
	JobType,
} from '../others/CommonComponents';
import { BASE_URL } from '../utils/constants';
import { useRouter } from 'next/navigation';

export const Card = ({ thisMonth, trend, Icon, title }: TCardProps) => {
	return (
		<div className='w-full h-full rounded-[8px] gap-[12px] flex'>
			<div className='flex-[3] flex flex-col justify-between p-[6px]'>
				<p className='font-bold leading-[24px] text-[16px] text-gray-body-text'>
					{title}
				</p>
				<div className='leading-[60px] text-[44px] font-bold text-title-text-color'>
					{thisMonth}
				</div>
				<div className='flex gap-[10px]'>
					<div className='flex gap-[2px]'>
						{trend ? (
							trend > 0 ? (
								<>
									<ArrowUpIcon
										{...{
											styles: {
												width: '24px',
												height: '24px',
											},
											svgElementClassName: 'stroke-fongreen-normal',
										}}
									/>
									<p className='text-fongreen-normal leading-[24px] text-[14px] font-normal'>
										{trend?.toFixed(2) ?? 0}%
									</p>
								</>
							) : (
								<>
									<ArrowUpIcon
										{...{
											styles: {
												width: '24px',
												height: '24px',
												rotate: '180deg',
											},
											svgElementClassName: 'stroke-job-closed-text-color',
										}}
									/>
									<p className='text-job-closed-text-color leading-[24px] text-[14px] font-normal'>
										{Math.abs(parseFloat(trend?.toFixed(2))) ?? 0}%
									</p>
								</>
							)
						) : (
							''
						)}
					</div>

					<p className='text-gray-body-text font-normal leading-[24px] text-p14px'>
						since last month
					</p>
				</div>
			</div>
			<div className='flex flex-col flex-[1] items-end pt-[4px] pr-[10px] justify-end'>
				{Icon ? (
					<Icon
						{...{
							styles: {
								height: '60px',
								width: '61px',
							},
							svgElementClassName: 'fill-foundation-dark-orange',
						}}
					/>
				) : null}
			</div>
		</div>
	);
};

export const DepartmentCard = ({ name, icon, id }: TMinistry) => {
	return (
		<div className='h-[84px] w-[286px] gap-[12px] rounded-[6px] p-[12px] border border-gray-border flex items-center justify-between bg-white'>
			<div className='flex-[1]'>
				{icon ? (
					<img
						src={icon.url}
						alt={icon.alternativeText}
						className='w-full h-auto'
					/>
				) : null}
			</div>

			<div className='flex flex-col gap-[6px] flex-[8]'>
				<p className='font-bold leading-[24px] tracking-[.5%] text-[16px] text-title-text-color'>
					{name}
				</p>
				<p className='font-normal leading-[24px] text-gray-body-text text-[14px]'>
					{id} open position
				</p>
			</div>
		</div>
	);
};

export const Comment = ({
	createdAt,
	message,
	user,
}: Pick<TCommentType, 'user' | 'message' | 'createdAt'>) => {
	const timePosted = formatDistanceStrict(new Date(createdAt), new Date(), {
		addSuffix: true,
	});

	return (
		<div className=''>
			<div className='flex justify-between'>
				<div className='flex gap-[6px] my-[6px]'>
					<p className='font-bold text-[16px] leading-[24px] text-name-color'>
						{user.username}
					</p>
					<span className='font-normal text-[14px] leading-[24px] text-gray-body-text'>
						says
					</span>
				</div>

				<div className='bg-delete-btn w-[30px] h-[30px] rounded-[8px] gap-[8px] flex items-center justify-center cursor-pointer selection:bg-inherit'>
					<DeleteIcon
						{...{
							svgElementClassName: 'stroke-red-500',
							applyToSvgEl: false,
							styles: {
								width: '20px',
								height: '20px',
							},
						}}
					/>
				</div>
			</div>

			<div className='my-[6px] selection:bg-inherit'>
				<div className='text-gray-body-text font-normal text-[14px] leading-[24px]'>
					{message}
				</div>

				<div className=''>
					<p className='text-green-text'>{timePosted} </p>
				</div>
			</div>
		</div>
	);
};

export const CommentCard = (comment: TCommentType) => {
	const timeLeft = formatDistanceStrict(
		new Date(),
		new Date(comment.job.application_end),
		{ addSuffix: true }
	);
	const router = useRouter();

	return (
		<div className='w-[96%] p-[20px] bg-white border py-[16px] mx-auto shadow-applicantBoxDetailsShadow rounded-[8px]'>
			<div className='flex gap-[16px]'>
				<p className='font-bold text-[18px] leading-[24.55px] text-title-text-color'>
					{comment?.job?.title}
				</p>

				<div className='w-fit'>
					<JobMinistryTag
						{...{
							ministry_name: comment.job?.ministry?.name,
							textClassName: 'text-dev-accent',
							className: 'bg-tag-color',
							dotClass: 'bg-dev-accent',
						}}
					/>
				</div>
			</div>

			<div className='my-[8px] flex gap-[24px] items-center'>
				<div className='w-fit'>
					<JobType {...{ name: comment.job?.job_type?.name }} />
				</div>

				<div className='w-fit'>
					<LocationLabel {...{ name: comment?.job?.location }} />
				</div>

				<div className='w-fit'>
					<TimeLimitLabel
						{...{
							time: timeLeft,
						}}
					/>
				</div>

				<div className='w-fit'>
					<ApplicantCountIcon {...{ count: comment.job?.applicationCount }} />
				</div>

				<div className='w-fit'>
					<div className='flex gap-[4px] items-center justify-center'>
						<MessageIcon
							{...{
								styles: {
									width: '20px',
									height: '20px',
								},
								svgElementClassName: 'stroke-gray-body-text',
							}}
						/>
						<p className='text-time-color'>
							{comment?.replies.length} comments
						</p>
					</div>
				</div>
			</div>

			<div className='my-[6px]'>
				<p className='text-gray-body-text font-bold text-[16px] leading-[24px]'>
					Comments
				</p>
				<div className='mb-[8px]'>
					{comment?.replies?.map(({ createdAt, message, user }, index) => (
						<Comment key={index} {...{ createdAt, message, user }} />
					))}
				</div>
				<div className='w-fit'>
					<div
						className='bg-hover-bg-color-btn w-fit h-fit flex gap-[2px] rounded-[8px] px-[14px] py-[8px] border items-center justify-center selection:bg-inherit shadow-btnBoxShadow'
						role='button'
						onClick={(e) =>
							router.push(`${BASE_URL}/jobs/${comment?.job?.id}`)
						}>
						<p className='text-dev-accent w-fit'>View Job</p>
						<ArrowRightIcon width={24} height={24} color='#0B7437' />
					</div>
				</div>
			</div>
		</div>
	);
};
