'use client';

import { DotsVerticalIcon } from '@radix-ui/react-icons';
import { ColumnDef } from '@tanstack/react-table';

import {
  ActionMapperType,
  Application,
  JobStatusEnum,
  TApplicantStatus,
  TColumnStaffDefinition,
  TJob,
  TMinistry,
  TUSER,
} from '../utils/types';
import { HelpIcon } from '../icons';
import Link from 'next/link';
import {
  DropDownWrapperCustomComponent,
  JobMinistryTag,
  JobType,
  Avatar,
  DropdownCustomComponentRadio,
} from '../others/CommonComponents';
import { DropdownMenuTrigger } from '../ui/dropdown-menu';
import { BASE_URL, JobTypes, UserOptionsEnum } from '../utils/constants';

export const JobsColumns: ColumnDef<TColumnStaffDefinition>[] = [
  {
    accessorKey: 'title',
    header: () => (
      <div className='w-full flex gap-[2px] items-center'>
        <p className='text-gray-body-text font-medium text-[12px] leading-[18px]'>
          Position
        </p>
        <HelpIcon
          {...{
            styles: { height: '16px', width: '16px' },
            svgElementClassName: 'stroke-dots-color',
          }}
        />
      </div>
    ),
    cell: ({ row }) => {
      const title: string = row.getValue('title');

      return (
        <div className='w-full flex'>
          <p className='m-auto'>{title}</p>
        </div>
      );
    },
  },
  {
    accessorKey: 'ministry',
    header: 'Department & Work Type',

    cell: ({ row }) => {
      //@ts-ignore
      const { ministry, job_type }: Pick<TJob, 'ministry' | 'job_type'> =
        row.original;

      return (
        <div className='w-full flex gap-[6px] items-center'>
          <div className='w-fit'>
            <JobMinistryTag
              {...{
                textClassName:
                  'text-gray-body-text leading-[16.37px] text-[12px] font-bold bg-gray-200',
                dotClass: 'bg-gray-body-text',
                ministry_name: ministry?.name,
                className: 'bg-gray-200',
              }}
            />
          </div>
          <div className='w-fit'>
            <JobType
              {...{
                className: `px-[12px] py-[4px] rounded-[40px] ${
                  job_type?.name === JobTypes.Attachment
                    ? 'text-brown-text border-attachment-border'
                    : job_type?.name === JobTypes.Contract
                    ? 'border-contract-border-color text-contract-text-color'
                    : 'text-brown-text border-attachment-border'
                }`,
                name: job_type?.name,
              }}
            />
          </div>
        </div>
      );
    },
  },
  {
    accessorKey: 'applicationCount',
    header: 'Candidates applied',
    cell: ({ row }) => {
      const count_applicants: number = row.getValue('applicationCount');

      return (
        <div className='h-fit flex  items-center md:w-fit md:rounded-[34px] md:py-[6px] md:px-[10px]'>
          {parseInt(`${count_applicants}`).toLocaleString('en-US')}
        </div>
      );
    },
  },

  {
    accessorKey: 'status',
    header: 'Status',
    cell: ({ row }) => {
      const status: string = row.getValue('status');

      return (
        <div className='flex'>
          <div
            className={`flex-grow-0 flex-shrink-0 flex-basis-auto h-[24px] w-fit leading-[20px] text-[12px] flex px-[8px] py-[2px] items-center rounded-[16px] font-semibold ${
              status === JobStatusEnum.Open
                ? 'bg-hover-bg-color-btn text-green-text'
                : status === JobStatusEnum.Closed
                ? 'bg-job-closed-bg-color text-job-closed-text-color'
                : status === JobStatusEnum.Awarded
                ? 'bg-job-awarded-bg-color text-job-awarded-text-color'
                : ''
            }`}>
            {status}
          </div>
        </div>
      );
    },
  },
  {
    accessorKey: 'creator',
    header: 'Created by',

    cell: ({ row }) => {
      const postedBy: TUSER = row.getValue('creator');

      return (
        <div className='w-full flex gap-[12px] items-center'>
          <Avatar
            {...{
              classNames: 'w-[40px] h-[40px]',
              name: postedBy?.username,
            }}
          />

          <p className='font-normal text-[14px] leading-[24px] text-gray-body-text'>
            {postedBy?.username}
          </p>
        </div>
      );
    },
  },
];

export const JobColumns: ColumnDef<TColumnStaffDefinition>[] = [
  {
    accessorKey: 'user',
    header: 'Name',
    cell: ({ row }) => {
      const user: TUSER = row.getValue('user');
      return (
        <div className='w-full flex gap-[12px] items-center'>
          <Avatar
            {...{
              classNames: 'w-[40px] h-[40px]',
              name: `${
                user?.firstname || user.lastname
                  ? `${user.firstname} ${user.lastname}`
                  : user.username
              }`,
            }}
          />
          <p className='font-normal text-[14px] leading-[24px] text-gray-body-text'>
            {user.firstname && user.lastname
              ? `${user.firstname} ${user.lastname}`
              : user?.username}
          </p>
        </div>
      );
    },
  },

  {
    accessorKey: 'status',
    header: 'Status',
    cell: ({ row }) => {
      const status: string = row.getValue('status');

      return (
        <div className='flex'>
          <div
            className={`flex-grow-0 flex-shrink-0 flex-basis-auto h-[24px] w-fit leading-[20px] text-[12px] flex px-[8px] py-[2px] items-center rounded-[16px] font-semibold ${
              status === TApplicantStatus.New
                ? 'bg-new-applicant-color text-new-applicant-text-color'
                : status === TApplicantStatus.Shortlisted ||
                  status === TApplicantStatus.Awarded
                ? 'bg-hover-bg-color-btn text-green-text'
                : status === TApplicantStatus.UnderReview
                ? 'bg-job-awarded-bg-color text-job-awarded-text-color'
                : status === TApplicantStatus.Rejected
                ? 'bg-job-closed-bg-color text-job-closed-text-color'
                : ''
            }`}>
            {status === TApplicantStatus.Shortlisted
              ? 'Shortlisted'
              : status === TApplicantStatus.Rejected
              ? 'Rejected'
              : status === TApplicantStatus.UnderReview
              ? 'Under review'
              : status === TApplicantStatus.Awarded
              ? 'Awarded'
              : status === TApplicantStatus.New
              ? 'New'
              : ''}
          </div>
        </div>
      );
    },
  },

  {
    accessorKey: 'user',
    header: 'Contact',
    cell: ({ row }) => {
      const contact: TUSER = row.getValue('user');

      return (
        <div className='h-fit flex flex-col  items-center px-[24px] py-[16px]'>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {contact.phone_number}
          </p>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {contact.email}
          </p>
        </div>
      );
    },
  },

  {
    accessorKey: 'createdAt',
    header: 'Date Applied',
    cell: ({ row }) => {
      const date_applied: string = row.getValue('createdAt');

      return (
        <div className='h-fit flex flex-col  items-center px-[24px] py-[16px]'>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {new Date(date_applied).toLocaleString('en-US', {
              day: '2-digit',
              month: 'short',
              year: 'numeric',
            })}
          </p>
        </div>
      );
    },
  },

  {
    accessorKey: 'id',
    header: () => 'Action',
    cell: ({ row }) => {
      const id = row.getValue('id');

      return (
        <div className='h-fit flex gap-[12px]  items-center px-[24px] py-[16px]'>
          <Link
            className='text-purple-800 font-bold text-[14px] leading-[24px]'
            href={`mailto:${id}`}>
            Email
          </Link>
          <Link href={`${BASE_URL}/applicants/${id}`}>
            <button className='border border-purple-400 shadow-btnBoxShadow rounded-[8px] px-[16px] py-[10px] font-bold leading-[24px]'>
              View
            </button>
          </Link>
        </div>
      );
    },
  },
];

export const ApplicantsColumns: ColumnDef<Application>[] = [
  {
    accessorKey: 'user',
    header: 'Name',

    cell: ({ row }) => {
      const user: TUSER = row.getValue('user');
      return (
        <div className='w-full flex gap-[12px] items-center'>
          <Avatar
            {...{
              classNames: 'w-[40px] h-[40px]',
              name: `${
                user?.firstname || user?.lastname
                  ? `${user?.firstname} ${user?.lastname}`
                  : user?.username
              }`,
            }}
          />
          {user && (
            <p className='font-normal text-[14px] leading-[24px] text-gray-body-text'>
              {user.firstname && user.lastname
                ? `${user.firstname} ${user.lastname}`
                : user?.username}
            </p>
          )}
        </div>
      );
    },
  },

  {
    accessorKey: 'status',
    header: 'Status',
    cell: ({ row }) => {
      const status: string = row.getValue('status');

      return (
        <div className='flex w-full'>
          <div
            className={`flex-grow-0 flex-shrink-0 flex-basis-auto h-[24px] w-fit leading-[20px] text-[12px] flex px-[8px] py-[2px] items-center rounded-[16px] font-semibold ${
              status === TApplicantStatus.New
                ? 'bg-new-applicant-color text-new-applicant-text-color'
                : status === TApplicantStatus.Shortlisted ||
                  status === TApplicantStatus.Awarded
                ? 'bg-hover-bg-color-btn text-green-text'
                : status === TApplicantStatus.UnderReview
                ? 'bg-job-awarded-bg-color text-job-awarded-text-color'
                : status === TApplicantStatus.Rejected
                ? 'bg-job-closed-bg-color text-job-closed-text-color'
                : ''
            }`}>
            {status === TApplicantStatus.Shortlisted
              ? 'Shortlisted'
              : status === TApplicantStatus.Rejected
              ? 'Rejected'
              : status === TApplicantStatus.UnderReview
              ? 'Under review'
              : status === TApplicantStatus.Awarded
              ? 'Awarded'
              : status === TApplicantStatus.New
              ? 'New'
              : ''}
          </div>
        </div>
      );
    },
  },
  {
    accessorKey: 'job',
    header: () => (
      <div className='w-full flex gap-[2px] items-center'>
        <p className='text-gray-body-text font-medium text-[12px] leading-[18px]'>
          Position applied
        </p>
        <HelpIcon
          {...{
            styles: { height: '16px', width: '16px' },
            svgElementClassName: 'stroke-dots-color',
          }}
        />
      </div>
    ),
    cell: ({ row }) => {
      const job: TJob = row.getValue('job');

      return (
        <div className='w-full flex'>
          <p className=''>{job?.title}</p>
        </div>
      );
    },
  },

  {
    accessorKey: 'job',
    header: 'Department & Work Type',

    cell: ({ row }) => {
      const job: TJob = row.getValue('job');

      return (
        <div className='w-full flex gap-[6px] items-center'>
          <div className='w-fit'>
            <JobMinistryTag
              {...{
                textClassName:
                  'text-gray-body-text leading-[16.37px] text-[12px] font-bold bg-gray-200',
                ministry_name: job?.ministry.name,
                dotClass: 'bg-gray-body-text',
                className: 'bg-gray-200',
              }}
            />
          </div>

          <div className='w-fit'>
            <JobType
              {...{
                className: `${
                  job.job_type?.name === JobTypes.Attachment
                    ? 'text-brown-text border-attachment-border'
                    : job.job_type?.name === JobTypes.Contract
                    ? 'border-contract-border-color text-contract-text-color'
                    : 'text-brown-text border-attachment-border'
                }`,
                name: job?.job_type?.name,
              }}
            />
          </div>
        </div>
      );
    },
  },
  {
    accessorKey: 'user',
    header: 'Contact',
    cell: ({ row }) => {
      const contact: TUSER = row.getValue('user');

      return (
        <div className='h-fit flex flex-col  items-center px-[24px] py-[16px]'>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {contact?.phone_number}
          </p>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {contact?.email}
          </p>
        </div>
      );
    },
  },

  {
    accessorKey: 'user',
    header: () => 'Action',
    cell: ({ row }) => {
      const user: TUSER = row.getValue('user');
      const { id } = row.original;

      return (
        <div className='h-fit flex gap-[12px]  items-center px-[24px] py-[16px]'>
          <Link
            className='text-purple-800 font-bold text-[14px] leading-[24px]'
            href={`mailto:${user?.email}`}>
            Email
          </Link>
          <Link href={`${BASE_URL}/applicants/${id}`}>
            <div className='border border-purple-400 shadow-btnBoxShadow rounded-[8px] px-[16px] py-[10px] font-bold leading-[24px]'>
              View
            </div>
          </Link>
        </div>
      );
    },
  },
];

export const StaffColumns: ColumnDef<TUSER>[] = [
  {
    accessorKey: 'id',
    header: 'Staff No',
  },
  {
    accessorKey: 'username',
    header: 'Name',

    cell: ({ row }) => {
      const name: string = row.getValue('username');

      return (
        <div className='w-full flex gap-[12px]'>
          {' '}
          <p className='font-normal text-[14px] leading-[24px] text-gray-body-text'>
            {name}
          </p>
        </div>
      );
    },
  },

  {
    accessorKey: 'staff_role',
    header: () => (
      <div className='w-full flex gap-[2px] items-center'>
        <p className='text-gray-body-text font-medium text-[12px] leading-[18px]'>
          Position
        </p>
      </div>
    ),
    cell: ({ row }) => {
      const position: string = row.getValue('staff_role');

      return (
        <div className='w-full flex'>
          <p className='font-normal text-[14px] leading-[24px] text-navy'>
            {position}
          </p>
        </div>
      );
    },
  },

  {
    accessorKey: 'ministry',
    header: 'Department',

    cell: ({ row }) => {
      const ministry: TMinistry = row.getValue('ministry');

      return (
        <div className='w-full flex gap-[6px] items-center'>
          <div className='w-fit'>
            <JobMinistryTag
              {...{
                textClassName:
                  'text-gray-body-text leading-[16.37px] text-[12px] font-bold bg-gray-200',
                ministry_name: ministry?.name ?? '',
                dotClass: 'bg-gray-body-text',
                className: 'bg-gray-200',
              }}
            />
          </div>
        </div>
      );
    },
  },
  {
    accessorKey: 'status',
    header: 'Status',
    cell: ({ row }) => {
      const user: TUSER = row.original;

      return (
        <div
          className={`h-[24px] w-fit leading-[20px] text-[12px] flex px-[8px] py-[2px] items-center rounded-[16px] font-semibold ${
            user.blocked || !user.confirmed
              ? 'bg-job-closed-bg-color text-job-closed-text-color'
              : 'bg-hover-bg-color-btn text-dev-accent'
          }`}>
          {user.blocked ? 'Suspended' : user.confirmed ? 'Active' : 'Inactive'}
        </div>
      );
    },
  },

  {
    accessorKey: 'phone_number',
    header: 'Contact',
    cell: ({ row }) => {
      const { phone_number, email }: Pick<TUSER, 'email' | 'phone_number'> =
        row.original;

      return (
        <div className='h-fit flex flex-col  items-center px-[24px] py-[16px]'>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {phone_number}
          </p>
          <p className='font-normal leading-[24px] text-[14px] text-gray-body-text'>
            {email}
          </p>
        </div>
      );
    },
  },
];

export const actionsColumn: ({
  ActionsHandlerMapping,
}: {
  ActionsHandlerMapping: ActionMapperType;
}) => ColumnDef<TColumnStaffDefinition> = ({ ActionsHandlerMapping }) => {
  return {
    accessorKey: 'id',
    header: 'Actions',
    enableHiding: false,
    cell: ({ row }) => {
      const id: undefined | number = row.getValue('id');

      return (
        <DropDownWrapperCustomComponent
          {...{
            components: (
              <DropdownMenuTrigger asChild>
                <div className='w-full flex items-center selection:bg-inherit'>
                  <div className='w-[20px] h-[20px] text-body-bg dark:text-white'>
                    <DotsVerticalIcon
                      width={20}
                      height={20}
                    />
                  </div>
                </div>
              </DropdownMenuTrigger>
            ),
          }}>
          <DropdownCustomComponentRadio
            {...{
              onChangeHandler: ActionsHandlerMapping.onChangeHandler,
              data: ActionsHandlerMapping.data,
              currPageSize: id,
            }}
          />
        </DropDownWrapperCustomComponent>
      );
    },
  };
};

export const userActionsColumn: ({
  ActionsHandlerMapping,
}: {
  ActionsHandlerMapping: ActionMapperType;
}) => ColumnDef<TUSER> = ({ ActionsHandlerMapping }) => {
  return {
    accessorKey: 'id',
    header: 'Actions',
    enableHiding: false,
    cell: ({ row }) => {
      const id: undefined | number = row.getValue('id');
      const { blocked } = row.original;

      return (
        <DropDownWrapperCustomComponent
          {...{
            components: (
              <DropdownMenuTrigger asChild>
                <div className='w-full flex items-center selection:bg-inherit'>
                  <div className='w-[20px] h-[20px] text-body-bg dark:text-white'>
                    <DotsVerticalIcon
                      width={20}
                      height={20}
                    />
                  </div>
                </div>
              </DropdownMenuTrigger>
            ),
          }}>
          <DropdownCustomComponentRadio
            {...{
              onChangeHandler: ActionsHandlerMapping.onChangeHandler,
              data: ActionsHandlerMapping.data.filter((item) =>
                item.label === UserOptionsEnum.Suspend && blocked
                  ? false
                  : item.label === UserOptionsEnum.Activate && !blocked
                  ? false
                  : true
              ),
              currPageSize: id,
            }}
          />
        </DropDownWrapperCustomComponent>
      );
    },
  };
};
