'use server';

import Strapi, { StrapiOptions, StrapiRequestParams } from 'strapi-sdk-js';
import { redirect } from 'next/navigation';
import { cookies } from 'next/headers';

import {
	Details,
	SERVER_ERROR,
	StrapiAuthenticationData,
	TApiHandlerProps,
	TError,
	TExpenses,
	TUSER,
} from './types';
import { BACKEND_URL, COOKIE_KEYS } from './constants';

export const deleteCookie = ({ name }: { name: string }) =>
	cookies().set(name, '', { maxAge: 0 });

export const getCookie = ({ name }: { name: string }) =>
	cookies().get(name)?.value;

export const getCookieAsync = async ({ name }: { name: string }) =>
	cookies().get(name)?.value;

export const getStrapiConfiguredInstance = (props?: Partial<StrapiOptions>) => {
	const auth = getCookie({ name: COOKIE_KEYS.auth });

	return new Strapi({
		url: BACKEND_URL,
		axiosOptions: {
			...(auth
				? {
						headers: {
							Authorization: `Bearer ${auth}`,
						},
				  }
				: {}),
		},
		...props,
	});
};

export const AuthenticateUser = async ({
	identifier,
	password,
}: StrapiAuthenticationData) => {
	const strapi = getStrapiConfiguredInstance();
	const response = strapi.login({ identifier, password });

	return response
		.then(({ user, jwt }) => {
			cookies().set(COOKIE_KEYS.auth, jwt, {
				// ...(process.env.NODE_ENV === 'production'
				// 	? {
				// 			domain: `${
				// 				new URL(process.env.NEXT_PUBLIC_HOSTED_URL!).hostname
				// 			}`,
				// 	  }
				// 	: { domain: 'localhost' }),
				httpOnly: true,
				maxAge: 30000,
				secure: true,
				path: '/',
			});

			cookies().set(COOKIE_KEYS.user, JSON.stringify(user), {
				// ...(process.env.NODE_ENV === 'production'
				// 	? {
				// 			domain: `${
				// 				new URL(process.env.NEXT_PUBLIC_HOSTED_URL!).hostname
				// 			}`,
				// 	  }
				// 	: { domain: 'localhost' }),
				httpOnly: false,
				maxAge: 30000,
				secure: true,
				path: '/',
			});

			return { data: user, err: null };
		})
		.catch((err: SERVER_ERROR) => ({
			data: null,
			err:
				err.error.status >= 400 && err.error.status <= 499
					? 'Invalid credentials provided'
					: 'Something went wrong.',
		}));
};

export const Logout = async () => {
	await getStrapiConfiguredInstance().logout();
	Object.values(COOKIE_KEYS).map((name) => deleteCookie({ name }));

	redirect('/');
};

export const fetchEndpointData = <dataTypeExpected = any>({
	options,
	url,
}: {
	options: StrapiRequestParams;
	url: string;
}) =>
	getStrapiConfiguredInstance().find<dataTypeExpected>(url, {
		...options,
	});

export const createResourceEndpointData = ({ data, url }: TApiHandlerProps) =>
	getStrapiConfiguredInstance()
		.create(url, data)
		.then((data: any) => ({ data, err: null }))
		.catch(({ error: { message, status, details } }: SERVER_ERROR) => ({
			err: { message, status, details },
			data: null,
		}));

export const updateResourceEndpointData = ({ data, url }: TApiHandlerProps) =>
	getStrapiConfiguredInstance()
		.axios({ method: 'PUT', data, url })
		.then(({ data }) => ({ data, err: null }))
		.catch(({ response: { status, data, message } }) => ({
			err: { status, details: data, message },
			data: null,
		}));

export const ResetPasswordHandler = ({ data, url }: TApiHandlerProps) =>
	getStrapiConfiguredInstance()
		.axios({ method: 'POST', data, url })
		.then(({ data }) => ({ data, err: null }))
		.catch(
			({
				response: { status, data, message },
			}: {
				response: TError & { data: Details };
			}) => ({
				err: { status, details: data, message },
				data: null,
			})
		);

export const deleteResourceEndpointData = ({ data, url }: TApiHandlerProps) =>
	getStrapiConfiguredInstance()
		.axios({ method: 'DELETE', data, url })
		.then(({ data }) => ({ data, err: null }))
		.catch(
			({
				response: { status, details: data, message },
			}: {
				response: TError;
			}) => ({
				err: { status, details: data, message },
				data: null,
			})
		);

export const fetchEndpointDataHomeTableWrapper = <TDatpeExpected>(
	data: TDatpeExpected
) => data;

export const fetchEndpointNetworkExpensesData = ({
	options,
	url,
}: {
	options: StrapiRequestParams;
	url: string;
}) =>
	getStrapiConfiguredInstance().find<TExpenses[]>(url, {
		...options,
	});

export const uploadResourceEndpointData = async ({
	method,
	data,
	url,
}: TApiHandlerProps) => {
	const auth = await getCookieAsync({ name: COOKIE_KEYS.auth });

	return fetch(`${BACKEND_URL}/api/${url}`, {
		body: data,
		method,
		headers: {
			Authorization: `Bearer ${auth}`,
		},
	})
		.then((data) => data.json())
		.then((data: any) => ({ data, err: null }))
		.catch(({ error: { message, status, details } }: SERVER_ERROR) => ({
			err: { message, status, details },
			data: null,
		}));
};

export const uploadFormDataEndpointData = async ({
	data,
	url,
}: TApiHandlerProps) => {
	const auth = await getCookieAsync({ name: COOKIE_KEYS.auth });

	return fetch(`${BACKEND_URL}/api/${url}`, {
		method: 'post',
		body: data,
		headers: {
			Authorization: `Bearer ${auth}`,
		},
	})
		.then((data) => data.json())
		.then((data: any) => ({ data, err: null }))
		.catch(({ error: { message, status, details } }: SERVER_ERROR) => ({
			err: { message, status, details },
			data: null,
		}));
};

export const updateCookie = ({
	updatedData,
	user,
}: {
	user: Partial<TUSER>;
	updatedData: { [key: string]: string };
}) => {
	cookies().set(COOKIE_KEYS.user, JSON.stringify({ ...user, ...updatedData }), {
		// ...(process.env.NODE_ENV === 'production'
		// 	? {
		// 			domain: `${
		// 				new URL(process.env.NEXT_PUBLIC_HOSTED_URL!).hostname
		// 			}`,
		// 	  }
		// 	: { domain: 'localhost' }),
		httpOnly: false,
		maxAge: 30000,
		secure: true,
		path: '/',
	});
};
